from argparse import ArgumentParser
from mail.unistat.cpp.cython.meters import (
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    SupervisorLogRestartMeters,
)
from collections import namedtuple
import os
import yaml
import logging
import mail.unistat.cpp.cython.logs as logs
import mail.barbet.unistat.cpp.run as unistat
from mail.ymod_queuedb_worker.unistat.cpp.metrics import WorkerMetricsLog, WorkerMetric


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/barbet/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('barbet_config')
    return parser.parse_args()


def remove_scheme(s):
    return s.replace('https://', '').replace('http://', '')


def remove_scheme_and_port(s):
    s = remove_scheme(s)
    i = s.rfind(':')
    if i != -1:
        return s[:i]
    return s


def get_module(data, name):
    res = list(filter(lambda m: m['_name'] == name, data['config']['modules']['module']))
    if not res:
        return {}
    return res[0]['configuration']


def make_barbet_config(data):
    worker = get_module(data, 'worker')
    ymod_s3 = get_module(data, 'ymod_s3')

    http_s3_log = ymod_s3.get('stats_logger', None)
    s3_host = None
    if http_s3_log:
        http_s3_log = data['config']['log'][http_s3_log]['sinks'][0]['path']
        http_s3_log = os.path.join(os.curdir, http_s3_log)

        s3_host = remove_scheme_and_port(ymod_s3['client']['endpoint_override'])

    return BarbetConfig(
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
        barbet_log=os.path.join(os.curdir, data['config']['log']['barbet']['sinks'][0]['path']),
        worker_log=os.path.join(os.curdir, data['config']['log']['worker_access']['sinks'][0]['path']),
        http_s3_log=http_s3_log,
        sharpei_host=remove_scheme_and_port(get_module(data, 'sharpei_client')['hosts']),
        blackbox_host=remove_scheme_and_port(worker['tasks']['fill_restore']['blackbox']['url']),
        nwsmtp_host=remove_scheme_and_port(worker['store']['url']),
        sendbernar_host=remove_scheme_and_port(worker['sendbernar']['url']),
        s3_host=s3_host,
    )


def make_http_meters(host):
    meters = [HttpClientHttpRequestCountByStatus(host, "http_client")]
    meters.append(HttpClientHttpRequestTotalTimeHist((0, 20, 50, 100, 300, 500), host, "http_client_" + host))
    return meters


def make_http_client_log_meters(cfg):
    meters = []

    for host in [cfg.sharpei_host, cfg.blackbox_host, cfg.nwsmtp_host, cfg.sendbernar_host]:
        meters.extend(make_http_meters(host))

    return meters


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def make_worker_log_meters():
    return [WorkerMetric()]


BarbetConfig = namedtuple('BarbetConfig', (
    'httpclient_log',
    'barbet_log',
    'worker_log',
    'http_s3_log',
    'sharpei_host',
    'blackbox_host',
    'nwsmtp_host',
    'sendbernar_host',
    's3_host',
))


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.barbet_config) as f:
        barbet_config = make_barbet_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    http_client_log = logs.HttpClientLog(
        [],
        make_http_client_log_meters(barbet_config),
        fast_forward,
        barbet_config.httpclient_log
    )

    worker_log = WorkerMetricsLog(
        [],
        make_worker_log_meters(),
        fast_forward,
        barbet_config.worker_log
    )

    logs_list = [http_client_log, worker_log]

    if barbet_config.http_s3_log:
        http_s3_log = logs.HttpClientLog(
            [],
            make_http_meters(barbet_config.s3_host),
            fast_forward,
            barbet_config.http_s3_log
        )
        logs_list.append(http_s3_log)

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
        logs_list.append(supervisor_log)

    unistat.run(args.host, args.port, logs_list, args.log, logLevel='info')

if __name__ == '__main__':
    main()
