#include <yplatform/find.h>
#include <yplatform/module_registration.h>
#include <yplatform/loader.h>

#include <mail/barbet/service/include/tasks/archive.h>
#include <mail/barbet/service/include/handlers/backup.h>
#include <mail/barbet/service/include/handlers/restore.h>
#include <mail/barbet/service/include/task_params.h>
#include <mail/barbet/service/include/log.h>

#include <mail/ymod_maildb/include/module.h>
#include <mail/http_getter/client/include/module.h>
#include <mail/http_getter/client/include/endpoint_reflection.h>
#include <yamail/data/deserialization/ptree_reader.h>
#include <mail/webmail/http_api_helpers/include/find_dependency.h>

#include <mail/ymod_queuedb_worker/include/module.h>
#include <mail/ymod_queuedb_worker/include/exec_or_wait.h>
#include <boost/asio/steady_timer.hpp>


namespace barbet {

struct Worker: public yplatform::module {
    void init(const yplatform::ptree& cfg) {
        using http_api::findDependency;

        const auto worker = findDependency<ymod_queuedb::WorkerModule>(cfg, "dependencies.worker");
        const auto maildb = findDependency<ymod_maildb::Module>(cfg, "dependencies.maildb");
        const auto getter = findDependency<http_getter::ClientModule>(cfg, "dependencies.http_getter");
        const auto reactor = findDependency<yplatform::reactor>(cfg, "dependencies.reactor");
        const auto s3 = findDependency<ymod_s3::Client>(cfg, "dependencies.s3");

        const ymod_queuedb::ExecOrWaitFactory factory(cfg.get_child("exec_or_wait"), reactor);

        const auto store = yamail::data::deserialization::fromPtree<http_getter::Endpoint>(
            cfg.get_child("store")
        );

        const auto blackbox = yamail::data::deserialization::fromPtree<http_getter::Endpoint>(
            cfg.get_child("tasks.fill_restore.blackbox")
        );

        const auto sendbernar = yamail::data::deserialization::fromPtree<http_getter::Endpoint>(
            cfg.get_child("sendbernar")
        );

        const auto messagesChunk = cfg.get<unsigned>("tasks.fill_restore.messages_chunk");
        const auto hiddenMessagesChunk = cfg.get<unsigned>("tasks.fill_restore.hidden_messages_chunk");
        const auto s3Bucket = cfg.get<std::string>("tasks.archive_restore.bucket");
        const auto archiveUpdateInterval = cfg.get<unsigned>("tasks.archive_restore.update_interval");
        

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.fill_backup"), fillBackupType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                ExecOrWait eow = factory.product(yield, ctx);
                FillBackupParams params = parseFillBackupParams(task, sendbernar, *maildb, *getter);
                return fillBackup(params, eow, lastTry, ctx, yield);
            }
        );

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.fill_restore"), fillRestoreType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                ExecOrWait eow = factory.product(yield, ctx);
                FillRestoreParams params = parseFillRestoreParams(task, messagesChunk,
                                                                  hiddenMessagesChunk,
                                                                  store, blackbox, sendbernar,
                                                                  *maildb, *getter);
                return fillRestore(params, eow, lastTry, ctx, yield);
            }
        );

        ymod_queuedb::addHandler(
            cfg.get_child("tasks.archive_restore"), archive::archiveRestoreType(), *worker,
            [=] (const ymod_queuedb::Task& task, bool lastTry, yplatform::task_context_ptr ctx, boost::asio::yield_context yield) {
                ExecOrWait eow = factory.product(yield, ctx);
                auto params = archive::parseRestoreTaskParams(task, *maildb, s3, s3Bucket, archiveUpdateInterval,
                                                        *getter, store, blackbox, sendbernar);
                return archive::archiveRestore(std::move(params), eow, lastTry, ctx, yield);
            }
        );

    }
};

}

DEFINE_SERVICE_OBJECT(barbet::Worker)
