#pragma once

#include <aws/core/Aws.h>
#include <aws/s3/S3Client.h>

#include <mail/barbet/ymod_s3/include/internal/io.h>

namespace ymod_s3 {

namespace s3m = Aws::S3::Model;

class Client {
public:
    virtual ~Client() = default;

    using OnGetObjectOutcome = io::Hook<s3m::GetObjectOutcome>;

    template<typename Handler = io::sync_context>
    auto GetObject(const s3m::GetObjectRequest& request, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnGetObjectOutcome> init(handler);
        asyncGetObject(request, [h = std::move(init.handler)](auto&& ec, auto& val) mutable {
            static_assert(!std::is_copy_constructible_v<std::decay_t<decltype(val.GetResult())>>, 
                        "extra lambda wrapper is no longer needed because <s3m::GetObjectResult> became copy constructible");
            h(std::forward<decltype(ec)>(ec), std::move(val));
        });
        return init.result.get();
    }

    using OnListObjectsV2Outcome = io::Hook<s3m::ListObjectsV2Outcome>;

    template<typename Handler = io::sync_context>
    auto ListObjectsV2(const s3m::ListObjectsV2Request& request, Handler handler = io::use_sync) const {
        io::detail::init_async_result<Handler, OnListObjectsV2Outcome> init(handler);
        asyncListObjectsV2(request, init.handler);
        return init.result.get();
    }


protected:
    virtual void asyncGetObject(const s3m::GetObjectRequest&, OnGetObjectOutcome) const = 0;

    virtual void asyncListObjectsV2(const s3m::ListObjectsV2Request&, OnListObjectsV2Outcome) const = 0;
};

using ClientPtr = std::shared_ptr<Client>;

}
