#pragma once

#include <aws/core/monitoring/DefaultMonitoring.h>
#include <logdog/logger.h>

namespace ymod_s3::detail {

namespace log {

LOGDOG_DEFINE_ATTRIBUTE(std::string, uri)
LOGDOG_DEFINE_ATTRIBUTE(unsigned, status)
LOGDOG_DEFINE_ATTRIBUTE(double, total_time)

using namespace ::logdog::attr;

constexpr static auto aws_http_formatter = ::logdog::tskv::make_formatter(BOOST_HANA_STRING("mail-aws_http-log"));

}

inline auto getRequestLogger(const std::string& logName) {
    auto logger = std::make_shared<yplatform::log::source>(YGLOBAL_LOG_SERVICE, logName);
    return ::logdog::make_log(log::aws_http_formatter, logger);
}

using RequestLogger = decltype(getRequestLogger(""));

class RequestMonitoring : public Aws::Monitoring::MonitoringInterface {

public:
    using CoreMetricsCollection = Aws::Monitoring::CoreMetricsCollection;

    explicit RequestMonitoring(RequestLogger logger) : logger_(std::move(logger)) {}

    void* OnRequestStarted(const Aws::String&, const Aws::String&,
        const std::shared_ptr<const Aws::Http::HttpRequest>&) const override {
        return nullptr;
    }

    void OnRequestSucceeded(const Aws::String&,
        const Aws::String& , const std::shared_ptr<const Aws::Http::HttpRequest>& request,
        const Aws::Client::HttpResponseOutcome& outcome, const CoreMetricsCollection& metricsFromCore, void*) const override {
        logRequest(request, outcome, metricsFromCore);
    }

    void OnRequestFailed(const Aws::String&,
        const Aws::String& , const std::shared_ptr<const Aws::Http::HttpRequest>& request,
        const Aws::Client::HttpResponseOutcome& outcome, const CoreMetricsCollection& metricsFromCore, void*) const override {
        logRequest(request, outcome, metricsFromCore);
    }

    void OnRequestRetry(const Aws::String& , const Aws::String& ,
        const std::shared_ptr<const Aws::Http::HttpRequest>& , void* ) const override { }

    void OnFinish(const Aws::String&, const Aws::String& ,
        const std::shared_ptr<const Aws::Http::HttpRequest>&, void* ) const override { }
private:

    void logRequest(const std::shared_ptr<const Aws::Http::HttpRequest>& request, const Aws::Client::HttpResponseOutcome& outcome, 
                const CoreMetricsCollection& metricsFromCore) const {
        using namespace Aws::Monitoring;
        static const Aws::String timeMetricName = GetHttpClientMetricNameByType(HttpClientMetricsType::RequestLatency);
        const Aws::Http::HttpResponseCode responseCode = outcome.IsSuccess() ? outcome.GetResult()->GetResponseCode()
                                                                             : outcome.GetError().GetResponseCode();
        double total_time = 0.;
        auto iter = metricsFromCore.httpClientMetrics.find(timeMetricName);
        if (iter != metricsFromCore.httpClientMetrics.end())
        {
            total_time = iter->second / 1000.0;
        }
    
        LOGDOG_(logger_, notice, 
            log::uri=request->GetURIString(true),
            log::status=static_cast<unsigned>(responseCode),
            log::total_time=total_time
        );
    }

    RequestLogger logger_;
};

class RequestMonitoringFactory : public Aws::Monitoring::MonitoringFactory {
public:
    explicit RequestMonitoringFactory(RequestLogger logger) : logger_(std::move(logger)) {}

    [[nodiscard]] Aws::UniquePtr<Aws::Monitoring::MonitoringInterface> CreateMonitoringInstance() const override {
        return Aws::MakeUnique<RequestMonitoring>(__PRETTY_FUNCTION__, logger_);
    }

private:
    RequestLogger logger_;
};

}

