#include <mail/barbet/ymod_s3/include/client.h>
#include <mail/barbet/ymod_s3/include/settings.h>
#include <mail/barbet/ymod_s3/include/internal/executor.h>
#include <mail/barbet/ymod_s3/include/internal/logger.h>
#include <mail/barbet/ymod_s3/include/internal/monitoring.h>

#include <yplatform/module_registration.h>
#include <yplatform/loader.h>
#include <ymod_tvm/module.h>
#include <yplatform/find.h>

#include <aws/core/auth/AWSCredentialsProvider.h>
#include <aws/s3/model/GetObjectRequest.h>
#include <aws/s3/model/ListObjectsV2Request.h>

#include <util/system/env.h>


namespace {

class TVMCredentialsProvider : public Aws::Auth::AWSCredentialsProvider
{
public:
    using TVMSettings = ymod_s3::Settings::Credentials::TVM;

    explicit TVMCredentialsProvider(TVMSettings tvmSettings) : tvmSettings_{std::move(tvmSettings)} {
        tvm_ = yplatform::find<ymod_tvm::tvm2_module, std::shared_ptr>(tvmSettings_.module);
        credentials_.SetAWSAccessKeyId(fmt::format("TVM_V2_{}", tvmSettings_.service_id));
        credentials_.SetAWSSecretKey("unused");
    }

    Aws::Auth::AWSCredentials GetAWSCredentials() final
    {
        std::string ticket;
        const auto err = tvm_->get_service_ticket(tvmSettings_.s3_id, ticket);
        if (err) {
            throw mail_errors::system_error(err);
        }
        credentials_.SetSessionToken(fmt::format("TVM2 {}", ticket));
        return credentials_;
    }

private:
    TVMSettings tvmSettings_;
    std::shared_ptr<ymod_tvm::tvm2_module> tvm_;
    Aws::Auth::AWSCredentials credentials_;
};



std::shared_ptr<Aws::Auth::AWSCredentialsProvider>
createCredentialsProvider(ymod_s3::Settings::Credentials::TVM& tvm) {
    return Aws::MakeShared<TVMCredentialsProvider>(__PRETTY_FUNCTION__, tvm);
}

std::shared_ptr<Aws::Auth::AWSCredentialsProvider>
createCredentialsProvider(ymod_s3::Settings::Credentials::Simple& cred) {
    return Aws::MakeShared<Aws::Auth::SimpleAWSCredentialsProvider>(__PRETTY_FUNCTION__, 
        cred.access_key_id, cred.secret_key, cred.session_token
    );
}

std::shared_ptr<Aws::Auth::AWSCredentialsProvider>
createCredentialsProvider(ymod_s3::Settings::Credentials::Credentials& cred) {
    std::shared_ptr<Aws::Auth::AWSCredentialsProvider> res;
    boost::fusion::for_each(cred, [&res](auto&& c) {
        if (!c) {
            return;
        }
        if (res) {
            throw std::invalid_argument("ambiguous credentials are specified");
        }
        res = createCredentialsProvider(c.value());
    });
    if (!res) {
        throw std::invalid_argument("credentials are not specified");
    }
    return res;
}


template<typename Handler>
auto s3AsyncHelper(Handler h) {
    return [h = std::move(h)](const Aws::S3::S3Client*, const auto& /*request*/, auto&& outcome,
                              const std::shared_ptr<const Aws::Client::AsyncCallerContext>&) {
        h(std::forward<decltype(outcome)>(outcome));
    };
}

void disableAwsEC2() {
    SetEnv("AWS_EC2_METADATA_DISABLED", "true");
}

}

namespace ymod_s3 {


namespace s3m = Aws::S3::Model;

class SimpleClient : public Client, public yplatform::module {
public:
    ~SimpleClient() override {
        Aws::ShutdownAPI(AWSOptions_);
    }

    explicit SimpleClient(yplatform::reactor& reactor, const yplatform::ptree& cfg = {}) {
        disableAwsEC2();
        
        auto settings = Settings::readSettings(cfg);

        AWSOptions_.loggingOptions.logLevel = settings.log_level;
        AWSOptions_.loggingOptions.logger_create_fn =
            [logName = settings.logger, logLevel = AWSOptions_.loggingOptions.logLevel]() {
                return Aws::MakeShared<detail::AwsLogger>(__PRETTY_FUNCTION__,
                                                            logName, logLevel
                );
            };

        if (!settings.stats_logger.empty()) {
            AWSOptions_
                .monitoringOptions.customizedMonitoringFactory_create_fn
                .emplace_back(
                    [logger=detail::getRequestLogger(settings.stats_logger)]() {
                        return Aws::MakeUnique<detail::RequestMonitoringFactory>(__PRETTY_FUNCTION__, logger);
                    }
                );
        }

        Aws::InitAPI(AWSOptions_);

        const auto& client = settings.client;
        Aws::Client::ClientConfiguration s3Config;
        s3Config.userAgent = client.user_agent;
        s3Config.scheme = client.scheme;
        s3Config.region = client.region.value_or(s3Config.region);
        s3Config.endpointOverride = client.endpoint_override;

        s3Config.connectTimeoutMs = client.connect_timeout_ms;
        s3Config.requestTimeoutMs = client.request_timeout_ms;
        s3Config.executor = Aws::MakeShared<detail::AwsExecutor>(__PRETTY_FUNCTION__, 
                                                                    yplatform::reactor::make_not_owning_copy(reactor));

        s3Config.caPath = client.ca_path.value_or("/etc/ssl/certs/");
        s3Config.caFile = client.ca_file.value_or(s3Config.caFile);
        s3Config.verifySSL = client.verify_ssl;        
        
        auto sign_policy = client.sign_payloads.value_or(Aws::Client::AWSAuthV4Signer::PayloadSigningPolicy::Never);
        bool use_virtual_addressing = client.use_virtual_addressing.value_or(false);
        
        std::shared_ptr credentialsProvider = createCredentialsProvider(settings.credentials);
        client_ = std::make_shared<decltype(client_)::element_type>(
                std::move(credentialsProvider), s3Config,
                sign_policy,
                use_virtual_addressing
        );
    }

private:
    void asyncGetObject(const s3m::GetObjectRequest& request, OnGetObjectOutcome handler) const override {
        client_->GetObjectAsync(request, s3AsyncHelper(std::move(handler)));
    }

    void asyncListObjectsV2(const s3m::ListObjectsV2Request& request, OnListObjectsV2Outcome handler) const override {
        client_->ListObjectsV2Async(request, s3AsyncHelper(std::move(handler)));
    }

    std::shared_ptr<Aws::S3::S3Client> client_;
    Aws::SDKOptions AWSOptions_;
};

}


DEFINE_SERVICE_OBJECT(ymod_s3::SimpleClient)
