from typing import Any, Optional, Type

from sendr_aiohttp.handler import BaseHandler as BHandler
from sendr_aiohttp.handler import BaseParser
from sendr_qlog import LoggerContext
from sendr_tvm.common import TicketCheckResult  # NOQA

from mail.beagle.beagle.api.exceptions import APIException
from mail.beagle.beagle.core.actions.base import BaseAction
from mail.beagle.beagle.core.exceptions import (
    BaseCoreError, CoreActionDenyError, CoreAlreadyExists, CoreDataError, CoreNotFoundError
)


class Parser(BaseParser):
    def handle_error(self, error, *args, **kwargs):
        raise APIException(code=400, message='Bad Request', params=error.messages)


class BaseHandler(BHandler):
    PARSER = Parser()

    @property
    def db_engine(self):
        return self.app.db_engine

    @property
    def logger(self) -> LoggerContext:
        return self.request['logger']

    @property
    def request_id(self) -> str:
        return self.request['request-id']

    @property
    def tvm(self) -> TicketCheckResult:
        return self.request['tvm']

    def get_user_uid(self) -> Optional[int]:
        return self.tvm.default_uid

    def _core_exception_result(self, exc: BaseCoreError) -> None:
        code = 500
        message = getattr(exc, 'message', 'Internal server error')
        params = getattr(exc, 'params', {})
        if isinstance(exc, CoreNotFoundError):
            code = 404
        elif isinstance(exc, CoreAlreadyExists):
            code = 409
        elif isinstance(exc, CoreActionDenyError):
            code = 403
        elif isinstance(exc, CoreDataError):
            code = 400
        raise APIException(code=code, message=message, params=params)

    async def run_action_setup(self, action_cls: Type[BaseAction], params: Optional[dict] = None) -> None:
        action_cls.context.logger = self.logger
        action_cls.context.request_id = self.request_id
        action_cls.context.db_engine = self.db_engine

    async def run_action(self, action_cls: Type[BaseAction], **kwargs: Any) -> Any:
        return await super().run_action(action_cls, kwargs)
