import sys
from typing import List, Optional

import click
from click import ClickException

from sendr_core.exceptions import BaseCoreError

from mail.beagle.beagle.core.actions.sync.sync_organization import (
    QueueSyncCurrentOrganizationAction, SyncCurrentOrganizationAction
)
from mail.beagle.beagle.core.exceptions import DirectoryOrgNotFoundError
from mail.beagle.beagle.utils.cli import action_command


@click.command()
@click.option('--org-id', type=int, default=None, help='Organization id')
@click.option('--stdin', is_flag=True, help='Read from stdin')
@click.option('--force', is_flag=True, help='Sync without checking revision')
@click.option('--run-async', is_flag=True, help='Async')
@action_command
async def cli(org_id: Optional[int] = None, stdin: bool = False, force: bool = False, run_async: bool = False) -> None:
    if (org_id is None) == (not stdin):
        raise ClickException('One option should be chosen: --org-id or --stdin')

    org_ids: List[int] = []
    if org_id is not None:
        org_ids.append(org_id)
    else:
        for line in sys.stdin:
            try:
                org_ids.append(int(line))
            except ValueError:
                raise ClickException(f'Invalid org_id: {line}')

    for org_id in org_ids:
        try:
            if run_async:
                await QueueSyncCurrentOrganizationAction(org_id=org_id, force=force).run()
                click.echo(f"Organization sync {org_id} was scheduled")
            else:
                await SyncCurrentOrganizationAction(org_id=org_id, force=force).run()
                click.echo(f"Organization {org_id} was synced")
        except BaseCoreError as exc:
            if isinstance(exc, DirectoryOrgNotFoundError):
                click.echo(f"Organization {org_id} not found", err=True)
            else:
                raise exc
