import sys
from typing import List, Optional

import click
from click import ClickException

from sendr_core.exceptions import BaseCoreError

from mail.beagle.beagle.core.actions.smtp_cache import GenerateSMTPCacheAction
from mail.beagle.beagle.storage import StorageContext
from mail.beagle.beagle.utils.cli import action_command


@click.command()
@click.option('--org-id', type=int, default=None, help='Smtp cache org_id')
@click.option('--stdin', is_flag=True, help='Read from stdin')
@click.option('--run-async', is_flag=True, help='Async')
@action_command
async def cli(org_id: Optional[int] = None, stdin: bool = False, run_async: bool = False) -> None:
    if (org_id is None) == (not stdin):
        raise ClickException('One option should be chosen: --org_id or --stdin')

    org_ids: List[int] = []
    if org_id is not None:
        org_ids.append(org_id)
    else:
        for line in sys.stdin:
            try:
                org_ids.append(int(line))
            except ValueError:
                raise ClickException(f'Invalid org_id: {line}')

    action = GenerateSMTPCacheAction
    async with StorageContext(action.context.db_engine, logger=action.logger) as storage:
        action.context.storage = storage  # type: ignore
        for org_id in org_ids:
            try:
                if run_async:
                    await GenerateSMTPCacheAction(org_id=org_id).run_async()
                    click.echo(f"Cache generation for {org_id} was scheduled")
                else:
                    await GenerateSMTPCacheAction(org_id=org_id).run()
                    click.echo(f"Cache for {org_id} was generated")
            except BaseCoreError as exc:
                raise exc
