from typing import Dict, Optional

from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.actions.smtp_cache import GenerateSMTPCacheAction
from mail.beagle.beagle.core.entities.enums import MailListType
from mail.beagle.beagle.core.entities.mail_list import MailList, MailListDescription
from mail.beagle.beagle.core.exceptions import DirectoryError, OrganizationNotFoundError, PassportError
from mail.beagle.beagle.interactions.directory import BaseDirectoryError
from mail.beagle.beagle.interactions.passport import BasePassportError
from mail.beagle.beagle.storage.exceptions import OrganizationNotFound


class CreateMailListAction(BaseDBAction):
    transact = True

    def __init__(self,
                 org_id: int,
                 username: str,
                 description: Optional[Dict[str, str]] = None,
                 uid: Optional[int] = None,
                 generate_cache: bool = False
                 ):
        super().__init__()
        self.org_id = org_id
        self.username = username
        self.description = description or {}
        self.uid: Optional[int] = uid
        self.generate_cache = generate_cache

    async def create_user(self) -> int:
        try:
            domain = await self.clients.directory.get_master_domain(self.org_id)
        except BaseDirectoryError as e:
            raise DirectoryError(params={'error': e.default_message})

        try:
            return await self.clients.passport.account_register_pdd(self.username, domain)
        except BasePassportError as e:
            raise PassportError(params={'error': e.default_message})

    async def handle(self) -> MailList:
        if self.uid is None:
            self.uid = await self.create_user()
            mail_list_type = MailListType.MANUAL
        else:
            mail_list_type = MailListType.AUTO
        assert self.uid is not None

        mail_list_entity = MailList(org_id=self.org_id,
                                    mail_list_type=mail_list_type,
                                    uid=self.uid,
                                    username=self.username,
                                    description=MailListDescription(**self.description))

        try:
            mail_list = await self.storage.mail_list.create(mail_list_entity)
        except OrganizationNotFound:
            raise OrganizationNotFoundError

        if self.generate_cache:
            await GenerateSMTPCacheAction(org_id=mail_list.org_id, mail_list_id=mail_list.mail_list_id).run()

        return mail_list
