from typing import Type

from sendr_utils import UserType, get_user_type

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.entities.smtp_cache import CacheResponse, Recipient, SMTPCache
from mail.beagle.beagle.core.exceptions import MailListNotFoundError
from mail.beagle.beagle.interactions.blackbox import UserInfo
from mail.beagle.beagle.interactions.blackbox.exceptions import BaseBlackboxError
from mail.beagle.beagle.storage.exceptions import SMTPCacheNotFound


class RecipientsAction(BaseDBAction):
    db_engine_name = 'sync'

    def __init__(self, email_to: str, email_from: str):
        super().__init__()
        self.email_to = email_to
        self.email_from = email_from

    async def _get_user_info(self, email: str, raise_: Type[Exception] = MailListNotFoundError) -> UserInfo:
        try:
            return await self.clients.blackbox.userinfo_by_login(email)
        except BaseBlackboxError:
            raise raise_

    async def handle(self) -> CacheResponse:
        user_info_to = await self._get_user_info(self.email_to)

        try:
            smtp_cache: SMTPCache = await self.storage.smtp_cache.get_by_uid(user_info_to.uid)
        except SMTPCacheNotFound:
            raise MailListNotFoundError

        recipients = []
        for index, subscription in enumerate(smtp_cache.value.subscriptions):
            if subscription.uid == user_info_to.uid and not settings.USE_DEFAULT_DOMAIN:
                recipients.append(Recipient(uid=subscription.uid, email=self.email_to))
            elif get_user_type(subscription.uid) == UserType.CONNECT:
                assert user_info_to.default_email
                domain = user_info_to.default_email.rsplit('@', 1)[1]
                recipients.append(Recipient(uid=subscription.uid, email=f'{subscription.local_part}@{domain}'))
            elif get_user_type(subscription.uid) == UserType.STAFF:
                recipients.append(Recipient(uid=subscription.uid, email=f'{subscription.local_part}@yandex-team.ru'))
            else:
                recipients.append(Recipient(uid=subscription.uid, email=f'{subscription.local_part}@yandex.ru'))

        # todo: здесь допиливаем приватные рассылки
        return CacheResponse(subscriptions=recipients)
