from typing import Any, Dict, Iterable, List, Optional, Set, cast

from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.entities.enums import SubscriptionType
from mail.beagle.beagle.core.entities.smtp_cache import CacheSubscription, CacheValue, SMTPCache
from mail.beagle.beagle.core.entities.unit import Unit
from mail.beagle.beagle.interactions.blackbox import UserInfo


class GenerateSMTPCacheAction(BaseDBAction):
    action_name = 'generate_smtp_cache'
    transact = True

    def __init__(self, org_id: int, mail_list_id: Optional[int] = None, uids: Optional[Iterable] = None):
        """
        :param org_id: id организации, для которой генерируем кеш
        :param mail_list_id: id рассылки, для которой генерируем кеш
        :param uids: массив uid'ов рассылок, для которых генерируем кеш
        Т.о. можно сгенерировать кеш как по mail_list_id, так и по uid'ам
        """
        super().__init__()
        self.org_id = org_id
        self.mail_list_id = mail_list_id
        self.uids = uids

    async def handle(self) -> None:
        async for mail_list in self.storage.mail_list.find(self.org_id, mail_list_id=self.mail_list_id, uids=self.uids):
            uids: Set[int] = {mail_list.uid}

            # UnitUnit

            kwargs: Dict[str, Any] = {
                'org_id': mail_list.org_id,
                'parent_unit_subscription_mail_list_id': mail_list.mail_list_id,
                'iterator': True
            }

            async for unit_unit in self.storage.unit_unit.find(**kwargs):
                assert isinstance(unit_unit.unit, Unit)
                if unit_unit.unit.uid is not None:
                    uids.add(unit_unit.unit.uid)

            # UnitUser

            kwargs = {
                'org_id': mail_list.org_id,
                'unit_subscription_mail_list_id': mail_list.mail_list_id,
                'iterator': True
            }

            async for unit_user in self.storage.unit_user.find(**kwargs):
                uids.add(unit_user.uid)

            # UserSubscription

            kwargs = {
                'org_id': mail_list.org_id,
                'mail_list_id': mail_list.mail_list_id,
                'subscription_type': SubscriptionType.INBOX,
                'iterator': True
            }

            async for user_subscription in self.storage.user_subscription.find(**kwargs):
                uids.add(user_subscription.uid)

            # Resolve uids in blackbox

            user_infos: List[UserInfo] = []
            if uids:
                user_infos = await self.clients.blackbox.userinfo_by_uids(uids)

            smtp_cache = SMTPCache(
                org_id=self.org_id,
                uid=mail_list.uid,
                value=CacheValue(
                    subscriptions=[
                        CacheSubscription(
                            uid=user_info.uid,
                            org_id=self.org_id,
                            local_part=cast(str, user_info.default_email).rsplit('@', 1)[0]
                        )
                        for user_info in user_infos
                    ]
                )
            )
            await self.storage.smtp_cache.create_or_update(smtp_cache)
