from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.actions.organization import CreateOrganizationAction, DeleteOrganizationAction
from mail.beagle.beagle.core.actions.smtp_cache import GenerateSMTPCacheAction
from mail.beagle.beagle.core.actions.sync.sync_unit_members import SyncUnitMembersAction
from mail.beagle.beagle.core.actions.sync.sync_units import SyncUnitsAction
from mail.beagle.beagle.core.actions.sync.sync_users import SyncUsersAction
from mail.beagle.beagle.core.entities.directory_organization import DirectoryOrganization
from mail.beagle.beagle.core.entities.enums import TaskType
from mail.beagle.beagle.core.entities.external_organization import BaseExternalOrganization
from mail.beagle.beagle.core.exceptions import DirectoryOrgNotFoundError
from mail.beagle.beagle.interactions.directory.exceptions import OrganizationDeletedError, UnknownOrganizationError
from mail.beagle.beagle.storage.exceptions import OrganizationNotFound


class SyncOrganizationAction(BaseDBAction):
    transact = True

    def __init__(self,
                 org_id: int,
                 external_organization: BaseExternalOrganization,
                 force: bool = False
                 ):
        super().__init__()
        self.org_id: int = org_id
        self.external_organization: BaseExternalOrganization = external_organization
        self.force: bool = force

    async def handle(self) -> None:
        self.logger.context_push(org_id=self.org_id)
        try:
            org = await self.storage.organization.get(self.org_id, skip_deleted=False)
            self.logger.context_push(current_revision=org.revision)
        except OrganizationNotFound:
            org = await CreateOrganizationAction(self.org_id).run()
        else:
            if org.is_deleted:
                self.logger.warning("Can't sync deleted org")
                return
            external_revision = await self.external_organization.get_revision()
            self.logger.context_push(next_revision=external_revision)
            if self.force or external_revision > org.revision:
                org.revision = external_revision
            else:
                self.logger.info('Organization is up to date.')
                return

        self.logger.info('Starting organization sync.')

        units, unit_affected_uids = await SyncUnitsAction(
            org_id=self.org_id,
            external_organization=self.external_organization,
        ).run()

        users, user_affected_uids = await SyncUsersAction(
            org_id=self.org_id,
            external_organization=self.external_organization,
        ).run()

        unit_member_affected_uids = await SyncUnitMembersAction(
            org_id=self.org_id,
            external_organization=self.external_organization,
            users=users,
            units=units,
        ).run()

        affected_uids = set.union(unit_affected_uids, user_affected_uids, unit_member_affected_uids)  # noqa
        self.logger.context_push(total_affected_uids=len(affected_uids))

        await self.storage.organization.save(org)
        await GenerateSMTPCacheAction(org_id=org.org_id, uids=list(affected_uids)).run()

        self.logger.info('Organization sync complete.')


class SyncDirectoryOrganizationAction(BaseDBAction):
    transact = True

    def __init__(self,
                 org_id: int,
                 external_organization: BaseExternalOrganization,
                 force: bool = False
                 ):
        super().__init__()
        self.org_id: int = org_id
        self.external_organization: BaseExternalOrganization = external_organization
        self.force: bool = force

    async def handle(self) -> None:
        try:
            await self.external_organization.get_revision()
        except OrganizationDeletedError:
            await DeleteOrganizationAction(org_id=self.org_id).run()
            self.logger.info(f'Organization {self.org_id} was deleted.')
        except UnknownOrganizationError:
            self.logger.warning(f'Unknown organization {self.org_id}')
            raise DirectoryOrgNotFoundError
        else:
            await SyncOrganizationAction(
                org_id=self.org_id,
                external_organization=self.external_organization,
                force=self.force,
            ).run()


class SyncCurrentOrganizationAction(BaseDBAction):
    transact = True
    action_name = 'sync_current_organization'

    def __init__(self, org_id: int, force: bool = False):
        super().__init__()
        self.org_id = org_id
        self.force = force
        self.external_organization = self._get_current_external_organization()

    def _get_current_external_organization(self) -> BaseExternalOrganization:
        current_organization_type = settings.CURRENT_ORGANIZATION_TYPE

        if current_organization_type == DirectoryOrganization.TYPE:
            return DirectoryOrganization(org_id=self.org_id, client=self.clients.directory)
        else:
            raise NotImplementedError('Incorrect settings.CURRENT_ORGANIZATION_TYPE')

    async def handle(self) -> None:
        if isinstance(self.external_organization, DirectoryOrganization):
            await SyncDirectoryOrganizationAction(
                org_id=self.org_id,
                external_organization=self.external_organization,
                force=self.force
            ).run()
        else:
            raise NotImplementedError('Incorrect settings.CURRENT_ORGANIZATION_TYPE')


class QueueSyncCurrentOrganizationAction(BaseDBAction):
    transact = True

    def __init__(self, org_id: int, force: bool = False):
        super().__init__()
        self.org_id = org_id
        self.force = force

    async def handle(self) -> None:
        self.logger.context_push(org_id=self.org_id, force=self.force)

        if self.org_id in settings.DIRECTORY_IGNORED_ORGANIZATIONS:
            self.logger.info(f'Ignore events from {self.org_id}')
            return

        try:
            org = await self.storage.organization.get(org_id=self.org_id, skip_deleted=False)
        except OrganizationNotFound:
            org = await CreateOrganizationAction(org_id=self.org_id).run()

        if org.is_deleted:
            self.logger.warning(f"Can't schedule sync for deleted org {self.org_id}")

        # Create task even when org is deleted. Otherwise we will not have anything but warning about sync attempt
        await self.storage.task.create(
            task_type=TaskType.SYNC_ORGANIZATION,
            org_id=self.org_id,
            params={'org_id': self.org_id, 'force': self.force}
        )
