from typing import Dict, List, Set, Tuple

from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.actions.unit.create import CreateUnitAction
from mail.beagle.beagle.core.actions.unit.delete import DeleteUnitAction
from mail.beagle.beagle.core.actions.unit.update import UpdateUnitAction
from mail.beagle.beagle.core.entities.external_organization import BaseExternalOrganization
from mail.beagle.beagle.core.entities.unit import Unit


class SyncUnitsAction(BaseDBAction):
    def __init__(self, org_id: int, external_organization: BaseExternalOrganization):
        super().__init__()
        self.org_id: int = org_id
        self.external_organization: BaseExternalOrganization = external_organization

    async def handle(self) -> Tuple[List[Unit], Set[int]]:
        external_units: Dict[Tuple[str, str], Unit] = {
            external_unit.external_key: external_unit
            async for external_unit in self.external_organization.get_units()
        }

        units = []
        total_affected_uids: Set[int] = set()

        units_for_update: List[Tuple[Unit, Unit]] = []
        async for unit in self.storage.unit.find(self.org_id):
            external_unit = external_units.pop(unit.external_key, None)
            affected_uids = {}
            if external_unit is None:
                affected_uids = await DeleteUnitAction(unit=unit).run()
                total_affected_uids.update(affected_uids)
            else:
                units_for_update.append((unit, external_unit))

        for unit, external_unit in units_for_update:
            unit, affected_uids = await UpdateUnitAction(unit=unit, unit_entity=external_unit).run()
            units.append(unit)
            total_affected_uids.update(affected_uids)

        for external_unit in external_units.values():
            unit, affected_uids = await CreateUnitAction(unit_entity=external_unit).run()
            units.append(unit)
            total_affected_uids.update(affected_uids)

        return units, total_affected_uids
