from typing import ClassVar, Optional, Set, Tuple

from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.actions.mail_list.delete import DeleteMailListAction
from mail.beagle.beagle.core.actions.unit.create_auto_mail_list import CreateAutoMailListUnitAction
from mail.beagle.beagle.core.actions.unit.get_dependent_uids import GetDependentUIDsUnitAction
from mail.beagle.beagle.core.entities.unit import Unit
from mail.beagle.beagle.core.exceptions import UnitUpdateError
from mail.beagle.beagle.storage.exceptions import MailListNotFound
from mail.beagle.beagle.utils.helpers import update_from_fields


class UpdateUnitAction(BaseDBAction):
    transact = True

    SOFT_UPDATE_FIELDS: ClassVar[Tuple[str, ...]] = ('name',)
    HARD_UPDATE_FIELDS: ClassVar[Tuple[str, ...]] = ('uid', 'username')

    def __init__(self, unit: Unit, unit_entity: Unit):
        super().__init__()
        self.unit: Unit = unit
        self.unit_entity: Unit = unit_entity

    async def update_mail_list(self, old_unit_uid: Optional[int]) -> Set[int]:
        if old_unit_uid == self.unit.uid:
            return set()

        if old_unit_uid is not None:
            try:
                mail_list = await self.storage.mail_list.get_by_uid(old_unit_uid)
                await DeleteMailListAction(mail_list=mail_list).run()
            except MailListNotFound:
                pass

        if self.unit.uid is None or self.unit.username is None:
            return set()

        try:
            mail_list = await self.storage.mail_list.get_by_uid(self.unit.uid)
            raise UnitUpdateError(f'Mail list {self.unit.uid} already exists.')
        except MailListNotFound:
            await CreateAutoMailListUnitAction(self.unit).run()

        return {self.unit.uid}

    async def handle(self) -> Tuple[Unit, Set[int]]:
        if self.unit.external_key != self.unit_entity.external_key:
            raise UnitUpdateError('External key does not match.')

        old_unit_uid = self.unit.uid

        affected_uids: Set[int] = set()
        soft_update = update_from_fields(self.unit, self.unit_entity, self.SOFT_UPDATE_FIELDS)
        hard_update = update_from_fields(self.unit, self.unit_entity, self.HARD_UPDATE_FIELDS)

        if soft_update or hard_update:
            self.unit = await self.storage.unit.save(self.unit)
        if hard_update:
            affected_uids.update(await self.update_mail_list(old_unit_uid))
            affected_uids.update(await GetDependentUIDsUnitAction(self.unit.org_id, self.unit.unit_id).run())

        return self.unit, affected_uids
