from typing import ClassVar, Tuple

from mail.beagle.beagle.core.actions.base import BaseDBAction
from mail.beagle.beagle.core.entities.user import User
from mail.beagle.beagle.core.exceptions import UserUpdateError
from mail.beagle.beagle.utils.helpers import update_from_fields


class UpdateUserAction(BaseDBAction):
    """
    Updates user with values from entity. User entity must have the same org_id, uid, username as user.
    Does not (and must not) affect mail lists.
    """
    transact = True

    SOFT_UPDATE_FIELDS: ClassVar[Tuple[str, ...]] = ('first_name', 'last_name')

    def __init__(self, user: User, user_entity: User):
        super().__init__()
        self.user: User = user
        self.user_entity: User = user_entity

    async def handle(self) -> User:
        self.logger.context_push(org_id=self.user.org_id, uid=self.user.uid)

        if any((
            self.user.org_id != self.user_entity.org_id,
            self.user.uid != self.user_entity.uid,
            self.user.username != self.user_entity.username,
        )):
            raise UserUpdateError(f"Can't update user with uid {self.user.uid}, org_id {self.user.org_id}")

        soft_update = update_from_fields(self.user, self.user_entity, self.SOFT_UPDATE_FIELDS)
        if soft_update:
            self.user = await self.storage.user.save(self.user)
            self.logger.debug('User updated.')

        return self.user
