from typing import Optional, Type

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.core.actions.mail_list.base import BaseMailListAction
from mail.beagle.beagle.core.actions.smtp_cache import GenerateSMTPCacheAction
from mail.beagle.beagle.core.actions.transact_email import TransactEmailAction
from mail.beagle.beagle.core.entities.enums import SubscriptionType
from mail.beagle.beagle.core.entities.task import Task
from mail.beagle.beagle.core.entities.user_subscription import UserSubscription
from mail.beagle.beagle.core.exceptions import UserNotFoundError, UserSubscriptionAlreadyExistsError
from mail.beagle.beagle.interactions.blackbox import UserInfo
from mail.beagle.beagle.interactions.blackbox.exceptions import BaseBlackboxError
from mail.beagle.beagle.storage.exceptions import UserNotFound, UserSubscriptionAlreadyExists


class CreateUserSubscriptionAction(BaseMailListAction):
    transact = True

    def __init__(self, org_id: int, mail_list_id: int, uid: int, subscription_type: SubscriptionType,
                 inviter_uid: Optional[int] = None):
        super().__init__(org_id=org_id, mail_list_id=mail_list_id)
        self.uid: int = uid
        self.subscription_type = subscription_type
        self.inviter_uid = inviter_uid

    async def _get_user_info(self, uid: int, raise_: Type[Exception] = UserNotFoundError) -> UserInfo:
        try:
            return await self.clients.blackbox.userinfo_by_uid(uid)
        except BaseBlackboxError:
            raise raise_

    async def _notify_user_on_subscription(self) -> Task:
        """Notifies the user of subscription."""
        assert self.mail_list
        render_context = {
            'mail_list_name': self.mail_list.username,
        }
        inviter = None
        if self.inviter_uid and self.inviter_uid != self.uid:
            try:
                assert self.org_id is not None
                inviter = await self.storage.user.get(uid=self.inviter_uid, org_id=self.org_id)
                render_context.update({'inviter_login': inviter.username,
                                       'inviter_first_name': inviter.first_name,
                                       'inviter_last_name': inviter.last_name})
            except UserNotFound:
                pass

        subscriber_info = await self._get_user_info(self.uid)
        to_email = subscriber_info.default_email
        mailing_id = settings.SENDER_MAILING_USER_SUBSCRIBED  # todo upd mailing list
        assert to_email and self.org_id
        return await TransactEmailAction(
            org_id=self.org_id,
            to_email=to_email,
            render_context=render_context,
            mailing_id=mailing_id
        ).run_async()

    async def handle(self) -> None:
        assert self.org_id and self.mail_list_id
        user_subscription_entity = UserSubscription(org_id=self.org_id,
                                                    mail_list_id=self.mail_list_id,
                                                    uid=self.uid,
                                                    subscription_type=self.subscription_type)
        try:
            await self.storage.user_subscription.create(user_subscription_entity)
            await self._notify_user_on_subscription()
        except UserNotFound:
            raise UserNotFoundError
        except UserSubscriptionAlreadyExists:
            raise UserSubscriptionAlreadyExistsError
        await GenerateSMTPCacheAction(org_id=self.org_id, mail_list_id=self.mail_list_id).run_async()
