from typing import Any, Dict, Iterable, List, Optional, Union

from aiohttp import ClientResponse  # type: ignore

from sendr_utils import split_list

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.interactions.base import BaseInteractionClient
from mail.beagle.beagle.interactions.blackbox.entities import UserInfo
from mail.beagle.beagle.interactions.blackbox.exceptions import (
    BlackBoxDefaultEmailNotFoundError, BlackBoxUserException, BlackBoxUserNotFound, UnknownBlackBoxException
)
from mail.beagle.beagle.utils.helpers import without_none

IntStr = Union[str, int]
UidType = Union[Iterable[IntStr], IntStr]


class BlackBoxClient(BaseInteractionClient[dict]):
    SERVICE = 'blackbox'
    BASE_URL = settings.BLACKBOX_API_URL.rstrip('/')
    TVM_ID = settings.BLACKBOX_TVM_ID

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> dict:
        if response.status >= 400:
            raise UnknownBlackBoxException(message='Unknown blackbox error', method=response.method)

        data = await response.json()

        if 'exception' in data:
            with self.logger:
                self.logger.context_push(exception=data['exception'])
                self.logger.error('Got exception from blackbox')

            raise UnknownBlackBoxException(
                message='Got exception from blackbox',
                params={'exception': data['exception']},
                method=response.method
            )

        return data

    def _default_email(self, user: Dict[str, Any]) -> str:
        emails = user.get('address-list') or []
        for email in emails:
            if email.get('default'):
                return email['address']
        raise BlackBoxDefaultEmailNotFoundError

    async def _userinfo(self,
                        uids: Optional[List[Union[str, int]]] = None,
                        login: Optional[str] = None,
                        userip: str = '127.0.0.1',
                        get_default_emails: Optional[bool] = True,
                        skip_exceptions: bool = True,
                        ) -> List[UserInfo]:
        result: List[UserInfo] = []

        params = without_none({
            'method': 'userinfo',
            'format': 'json',
            'userip': userip,
            'login': login,
            'uid': ','.join(map(str, uids)) if uids else None,
            'emails': 'getdefault' if get_default_emails else None,
        })

        data: Dict[str, Any] = await self.get('userinfo', self.BASE_URL, params=params)

        for user in data['users']:
            self.logger.context_push(id=user.get('id'))

            if 'exception' in user:
                if skip_exceptions:
                    with self.logger:
                        self.logger.context_push(exception=user['exception'].get('value'))
                        self.logger.error('Blackbox userinfo error')
                else:
                    raise BlackBoxUserException(method='userinfo', params=user['exception'])

            try:
                uid = user['id']
                if not uid:
                    raise BlackBoxUserNotFound
                uid = int(uid)
                result.append(
                    UserInfo(
                        uid=uid,
                        default_email=self._default_email(user) if get_default_emails else None
                    )
                )
            except BlackBoxDefaultEmailNotFoundError as e:
                if skip_exceptions:
                    self.logger.warning('Blackbox default email in userinfo error')
                else:
                    raise e

        return result

    async def userinfo_by_uids(self,
                               uids: Iterable[Union[str, int]],
                               userip: str = '127.0.0.1',
                               get_default_emails: Optional[bool] = True,
                               skip_exceptions: bool = True,
                               chunk_size: int = 100) -> List[UserInfo]:
        result: List[UserInfo] = []

        for uids_chunk in split_list(list(uids), size=chunk_size):
            result += await self._userinfo(uids=uids_chunk,
                                           userip=userip,
                                           get_default_emails=get_default_emails,
                                           skip_exceptions=skip_exceptions)

        return result

    async def userinfo_by_uid(self,
                              uid: int,
                              userip: str = '127.0.0.1',
                              get_default_emails: Optional[bool] = True) -> UserInfo:
        userinfos = await self.userinfo_by_uids(uids=[uid],
                                                userip=userip,
                                                get_default_emails=get_default_emails,
                                                skip_exceptions=False)
        return userinfos[0]

    async def userinfo_by_login(self,
                                login: str,
                                userip: str = '127.0.0.1',
                                get_default_emails: Optional[bool] = True) -> UserInfo:

        result = await self._userinfo(login=login,
                                      userip=userip,
                                      get_default_emails=get_default_emails,
                                      skip_exceptions=False)
        return result[0]
