from typing import AsyncIterable, ClassVar, List, Tuple

from aiohttp import ClientResponse  # type: ignore

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.interactions.base import BaseInteractionClient
from mail.beagle.beagle.interactions.directory.entities import (
    DirectoryDepartment, DirectoryGroup, DirectoryObjectType, DirectoryUser
)
from mail.beagle.beagle.interactions.directory.exceptions import BaseDirectoryError, InvalidMasterDomainError


class DirectoryClient(BaseInteractionClient[dict]):
    SERVICE = 'directory'
    BASE_URL = settings.DIRECTORY_API_URL.rstrip('/')
    TVM_ID = settings.DIRECTORY_TVM_ID

    _max_per_page: ClassVar[int] = settings.DIRECTORY_MAX_PER_PAGE

    @classmethod
    async def _handle_response_error(cls, response: ClientResponse) -> None:
        response_data = await response.json()
        error_code = response_data.get('code')
        exc_cls = BaseDirectoryError.get_exception_by_code(error_code)
        raise exc_cls(method=response.method, message=error_code)

    async def _make_request(self, interaction_method: str, method: str, url: str,  # type: ignore
                            **kwargs: dict) -> dict:
        org_id = kwargs.pop('org_id', None)
        if org_id:
            kwargs.setdefault('headers', {})
            kwargs['headers']['X-ORG-ID'] = str(org_id)
        return await super()._make_request(interaction_method, method, url, **kwargs)

    async def _get_collection(self, interaction_method: str, url: str, **kwargs) -> AsyncIterable[dict]:  # type: ignore
        interaction_method += '/_get_collection'
        while url:
            response = await self.get(interaction_method, url, **kwargs)
            for result in response['result']:
                yield result
            url = response['links'].get('next')

    async def get_departments(self, org_id: int) -> AsyncIterable[DirectoryDepartment]:
        collection = self._get_collection(
            'get_departments',
            self.endpoint_url('v11/departments/'),
            org_id=org_id,
            params={
                'per_page': self._max_per_page,
                'fields': ','.join((
                    'id',
                    'name',
                    'label',
                    'uid',
                    'parent_id',
                )),
            },
        )
        async for department_json in collection:
            yield DirectoryDepartment(
                department_id=department_json['id'],
                name=department_json['name'],
                label=department_json['label'],
                uid=department_json['uid'],
                parent_id=department_json['parent_id'],
            )

    async def get_groups(self, org_id: int) -> AsyncIterable[DirectoryGroup]:
        collection = self._get_collection(
            'get_groups',
            self.endpoint_url('v11/groups/'),
            org_id=org_id,
            params={
                'per_page': self._max_per_page,
                'fields': ','.join((
                    'id',
                    'name',
                    'label',
                    'uid',
                )),
            }
        )
        async for group_json in collection:
            yield DirectoryGroup(
                group_id=group_json['id'],
                name=group_json['name'],
                label=group_json['label'],
                uid=group_json['uid'],
            )

    async def get_group_members(self, org_id: int, group_id: int) -> AsyncIterable[Tuple[DirectoryObjectType, int]]:
        members = await self.get(
            'get_group_members',
            self.endpoint_url(f'v11/groups/{group_id}/members/'),
            org_id=org_id,
            params={'per_page': self._max_per_page}
        )
        for member_json in members:
            try:
                object_type = DirectoryObjectType(member_json['type'])
            except ValueError:
                with self.logger:
                    self.logger.context_push(object_type=member_json['type'])
                    self.logger.exception('Unknown directory object type.')
                continue
            yield (object_type, member_json['object']['id'])

    async def get_organization_revision(self, org_id: int) -> int:
        response = await self.get(
            'get_organization_revision',
            self.endpoint_url(f'v11/organizations/{org_id}/'),
            params={'fields': 'revision'},
        )
        return response['revision']

    async def get_users(self, org_id: int) -> AsyncIterable[DirectoryUser]:
        collection = self._get_collection(
            'get_users',
            self.endpoint_url('v11/users/'),
            org_id=org_id,
            params={
                'per_page': self._max_per_page,
                'fields': ','.join((
                    'id',
                    'name',
                    'email',
                    'departments',
                    'groups',
                )),
            }
        )
        async for user_json in collection:
            local_part = user_json['email'].rsplit('@', 1)[0]
            yield DirectoryUser(
                user_id=user_json['id'],
                first_name=user_json['name'].get('first', ''),
                last_name=user_json['name'].get('last', ''),
                local_part=local_part,
                departments=[department_json['id'] for department_json in user_json['departments']],
                groups=[group_json['id'] for group_json in user_json['groups']],
            )

    async def get_master_domain(self, org_id: int) -> str:
        domains: List[dict] = await self.get(  # type: ignore
            'get_master_domain',
            self.endpoint_url('v11/domains/'),
            org_id=org_id,
            params={'fields': 'master'}
        )

        domains = [domain for domain in domains if domain.get('master')]
        if len(domains) != 1:
            raise InvalidMasterDomainError(method='get', message='There isn\'t 1 domains with master is true')

        return domains[0]['name']
