from typing import ClassVar, Dict, Type

from sendr_interactions.exceptions import BaseInteractionError


class BaseDirectoryError(BaseInteractionError):
    _exceptions_by_code: ClassVar[Dict[str, Type['BaseDirectoryError']]] = {}

    def __init__(self, *, method, message, params=None):
        if params is not None:
            self.params = params
        super().__init__(service='directory', method=method, message=message)

    def __init_subclass__(cls) -> None:
        code = getattr(cls, 'CODE', None)
        if code is not None:
            assert code not in BaseDirectoryError._exceptions_by_code
            BaseDirectoryError._exceptions_by_code[code] = cls

    @classmethod
    def get_exception_by_code(cls, code: str) -> Type['BaseDirectoryError']:
        return cls._exceptions_by_code.get(code, UnknownDirectoryError)


class UnknownDirectoryError(BaseDirectoryError):
    pass


class InvalidMasterDomainError(BaseDirectoryError):
    default_message = 'INVALID_MASTER_DOMAIN'
    CODE = 'invalid_master_domain'


class BaseNotFoundError(BaseDirectoryError):
    pass


class NotFoundError(BaseNotFoundError):
    CODE = 'not_found'


class OrganizationNotFoundError(BaseNotFoundError):
    pass


class OrganizationDeletedError(OrganizationNotFoundError):
    CODE = 'organization_deleted'


class UnknownOrganizationError(OrganizationNotFoundError):
    CODE = 'unknown_organization'
