from typing import List, Optional

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.interactions import BaseInteractionClient
from mail.beagle.beagle.interactions.hound.entities import Envelope
from mail.beagle.beagle.utils.helpers import without_none


class HoundClient(BaseInteractionClient[dict]):
    # https://wiki.yandex-team.ru/users/jkennedy/ywmiapi/
    SERVICE = 'hound'
    BASE_URL = settings.HOUND_API_URL.rstrip('/')
    TVM_ID = settings.HOUND_TVM_ID

    def _process_envelope(self, data: dict) -> Envelope:
        return Envelope(
            mid=int(data['mid']),
            fid=int(data['fid']),
            tid=int(data['threadId']),
            revision=data['revision'],
        )

    async def get_threads_by_folder(self, uid: int, fid: int, limit: int = 100, offset: int = 0) -> List[Envelope]:
        response = await self.get(
            'threads_by_folder',
            self.endpoint_url('threads_by_folder'),
            params={'uid': uid, 'fid': fid, 'first': offset, 'count': limit}
        )

        return [
            self._process_envelope(envelope)
            for envelope in response['threads_by_folder']['envelopes']
        ]

    async def get_messages_by_thread(self, uid: int, tid: int, limit: int = 100, offset: int = 0,
                                     sort_type: Optional[str] = None) -> List[Envelope]:
        response = await self.get(
            'messages_by_thread',
            self.endpoint_url('messages_by_thread'),
            params=without_none({'uid': uid, 'tid': tid, 'first': offset, 'count': limit, 'sort_type': sort_type})
        )

        return [
            self._process_envelope(envelope)
            for envelope in response['envelopes']
        ]

    async def get_attachment_sid(self, uid: int, mid: int, hid: str) -> str:
        sids_response = await self.post(
            'attach_sid',
            self.endpoint_url('attach_sid'),
            json=[{'uid': uid, 'downloads': [{'mid': mid, 'hids': [hid]}]}]
        )
        return sids_response['result'][0]['sids'][0]
