import base64
from enum import Enum, unique
from typing import Any, Dict, Iterable, Optional

from aiohttp.client import ClientResponse, ContentTypeError

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.interactions.base import BaseInteractionClient
from mail.beagle.beagle.interactions.mbody.exceptions import MBodyError, MidNotFoundMBodyError, UidNotFoundMBodyError


@unique
class MessageEndpointFlag(Enum):
    """Allowed flag values for /message endpoint"""
    IGNORE_HIGHLIGHT = 'IgnoreHighlight'
    OUTPUT_AS_CDATA = 'OutputAsCDATA'
    SHOW_CONTENT_META = 'ShowContentMeta'
    CUT_SIGNATURE = 'CutSignature'
    XML_STREAMER_ON = 'XmlStreamerOn'
    XML_STREAMER_MOBILE = 'XmlStreamerMobile'
    NO_VDIRECT_LINKS_WRAP = 'noVdirectLinksWrap'
    SEQUENTIAL_FACTEX = 'SequentialFactex'
    DRAFT = 'Draft'
    NO_TRIM = 'NoTrim'


class MBodyClient(BaseInteractionClient[dict]):
    """https://wiki.yandex-team.ru/pochta/backend/message-body"""
    SERVICE = 'mbody'
    BASE_URL = settings.MBODY_API_URL.rstrip('/')
    TVM_ID = settings.MBODY_TVM_ID

    _MESSAGE_SOURCE_XML_FIRST_LINE = b'<?xml version="1.0" encoding="utf-8"?>'
    _MESSAGE_SOURCE_XML_LAST_LINE = b'</message>'

    def _no_such_uid(self, response_error: str) -> bool:
        if 'uid not found' in response_error:
            # message and message_source endpoints
            return True
        if 'Sharpei service responded with 404' in response_error:
            # headers endpoint
            return True
        return False

    def _no_such_mid(self, response_error: str) -> bool:
        if 'unknown mid' in response_error:
            # message and message_source endpoints
            return True
        if 'returns records count not equal mids count' in response_error:
            return True
        return False

    async def _handle_response_error(self, response: ClientResponse) -> None:
        err_cls = MBodyError

        if response.status == 500:
            try:
                data = await response.json()
            except ContentTypeError:
                content = await response.text()
                with self.logger:
                    self.logger.context_push(content=content)
                    self.logger.error('MBody decode failed.')

                raise MBodyError(
                    status_code=response.status,
                    service=self.SERVICE,
                    method=response.method,
                    message='MBody response decode fail.'
                )

            error = data.get('error')
            if isinstance(error, str):
                if self._no_such_mid(error):
                    err_cls = MidNotFoundMBodyError
                elif self._no_such_uid(error):
                    err_cls = UidNotFoundMBodyError

        raise err_cls(
            status_code=response.status,
            service=self.SERVICE,
            method=response.method,
        )

    async def _is_failed(
        self,
        response: ClientResponse,
    ) -> bool:
        inherited_failed = await super()._is_failed(response)
        if inherited_failed:
            try:
                await self._handle_response_error(response)
            except (MidNotFoundMBodyError, UidNotFoundMBodyError):
                return False

        return inherited_failed

    def _clean_message_source_xml_header_if_present(self, message_source: bytes) -> bytes:
        first_header_line = self._MESSAGE_SOURCE_XML_FIRST_LINE
        last_header_line = self._MESSAGE_SOURCE_XML_LAST_LINE

        if message_source.startswith(first_header_line):
            last_line_start_index = message_source.find(last_header_line)
            if last_line_start_index == -1:
                self.logger.context_push(xml_header_first_line=first_header_line, xml_header_last_line=last_header_line)
                self.logger.error('Bad message source: found XML header first line, but unable to find last line')
                return b''
            else:
                header_stop_index = last_line_start_index + len(last_header_line) + 1
                while header_stop_index < len(message_source) and message_source[header_stop_index] == b'\n':
                    header_stop_index += 1
                return message_source[header_stop_index:]
        else:
            return message_source

    async def message(self, uid: int, mid: int, flags: Optional[Iterable[MessageEndpointFlag]] = None) -> dict:
        url = self.endpoint_url('message')
        params: Dict[str, Any] = {
            'uid': uid,
            'mid': mid,
        }
        if flags is not None:
            params['flags'] = ','.join(f.value for f in flags)

        return await self.get('message', url, params=params)  # type: ignore

    async def message_content(self, uid: int, mid: int, flags: Optional[Iterable[MessageEndpointFlag]] = None) -> str:
        message = await self.message(uid, mid, flags)
        # https://github.yandex-team.ru/tools/ml/blob/master/src/maillists/templates/mailarchive/message/body.html#L6
        # Сейчас берется просто первое тело письма
        # Беседовал со специалистами по mbody, фронт своими сложными законами определяет что показывать,
        # Лицензионный способ показать нужный вариант письма на данный момент отсутствует
        bodies = [body for body in message['bodies'] if not body['isAttach']]
        return bodies[0]['transformerResult']['textTransformerResult']['content']

    async def message_source(self, uid: int, mid: int) -> bytes:
        with self.logger as logger:
            logger.context_push(uid=uid, mid=mid)
            data = await self.get(
                'message_source',
                self.endpoint_url('message_source'),
                params={'uid': uid, 'mid': mid}
            )
            source = base64.decodebytes(data['text'].encode('utf8'))
            return self._clean_message_source_xml_header_if_present(source)
