from aiohttp import ClientResponse  # type: ignore

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.interactions.base import BaseInteractionClient
from mail.beagle.beagle.interactions.passport.exceptions import BasePassportError, UnknownPassportError


class PassportClient(BaseInteractionClient[dict]):
    SERVICE = 'passport'
    PASSPORT_TRUE = str(True).lower()
    PASSPORT_FALSE = str(False).lower()

    BASE_URL = settings.PASSPORT_API_URL.rstrip('/')
    TVM_ID = settings.PASSPORT_TVM_ID

    async def _process_response(self, response: ClientResponse, interaction_method: str) -> dict:
        if response.status >= 400:
            raise UnknownPassportError(message='Unknown passport error', method=response.method)

        data = await response.json()

        if 'status' not in data or data['status'] != 'ok':
            errors = data.get('errors', []) or []
            self.logger.context_push(errors=errors, status=data.get('status'))
            self.logger.error('Got not "ok" from passport')

            exc_cls = BasePassportError.get_exception_by_error(errors[-1] if len(errors) == 1 else None)

            raise exc_cls(
                message='Got status "error" from passport',
                params={'errors': errors},
                method=response.method
            )

        return data

    async def _make_request(self, *args, **kwargs):
        params = kwargs.pop('params', {})
        headers = kwargs.pop('headers', {})

        params['consumer'] = settings.PASSPORT_CONSUMER
        headers['Ya-Consumer-Client-Ip'] = self.ip

        kwargs['params'] = params
        kwargs['headers'] = headers

        return await super()._make_request(*args, **kwargs)

    # https://wiki.yandex-team.ru/passport/api/bundle/registration/pdd/
    async def account_register_pdd(self, login: str, domain: str) -> int:
        url = self.endpoint_url('1/bundle/account/register/pdd/')
        data = {
            'login': login,
            'domain': domain,
            'no_password': self.PASSPORT_TRUE,
            'is_maillist': self.PASSPORT_TRUE,
        }

        res = await self.post('account_register_pdd', url, data=data)
        return res['uid']
