from typing import ClassVar, Optional

import ujson
from aiohttp import BasicAuth

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.interactions.base import BaseInteractionClient


class SenderClient(BaseInteractionClient[dict]):
    """Клиент для запросов в рассылятор."""
    SERVICE = 'sender'
    BASE_URL = settings.SENDER_API_URL.strip('/')
    ACCOUNT_SLUG: ClassVar[str] = settings.SENDER_CLIENT_ACCOUNT_SLUG
    TOKEN: ClassVar[str] = settings.SENDER_CLIENT_TOKEN

    def _get_session_kwargs(self) -> dict:
        """Аргументы для конструктора объекта HTTP-сессии."""
        kwargs = super()._get_session_kwargs()
        kwargs['auth'] = BasicAuth(login=self.TOKEN)
        return kwargs

    async def send_transactional_letter(self,
                                        mailing_id: str,
                                        to_email: str,
                                        render_context: Optional[dict] = None,
                                        reply_email: Optional[str] = None) -> str:
        """Отправить письмо транзакционной рассылки в рассыляторе на указанный email."""
        url = self.endpoint_url(f'api/0/{self.ACCOUNT_SLUG}/transactional/{mailing_id}/send')
        params = {'to_email': to_email}
        json = {
            "async": True,
            "args": ujson.dumps(render_context or {}),
        }

        headers = dict()
        if reply_email:
            headers['Reply-To'] = reply_email
        if headers:
            json['headers'] = ujson.dumps(headers)

        resp_json = await self.post(
            interaction_method='send_transactional_letter',
            url=url,
            headers={},
            json=json,
            params=params
        )

        return resp_json['result']['message_id']
