import asyncio
import random
from collections import defaultdict
from typing import Callable, Dict, Optional, Set

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.core.exceptions import CoreDataError
from mail.beagle.beagle.mocks.utils import randdomain, randn, rands

ORGS: dict = {}
BB: dict = defaultdict(dict)

locks: Dict[int, asyncio.Lock] = defaultdict(asyncio.Lock)


def make_uid_generator() -> Callable:
    issued_uids: Set[int] = set()

    def _inner() -> int:
        while True:
            # https://wiki.yandex-team.ru/passport/uids/
            uid = 1130
            for _ in range(13):
                uid = (uid * 10) + randn(min=0, max=9)

            if uid not in issued_uids:
                issued_uids.add(uid)
                return uid

    return _inner


uid_generator = make_uid_generator()


def _create_groups(org_id: int, domain: str) -> list:
    groups: list = []
    for group_id in range(1, settings.MOCK_DIRECTORY_GROUP_AMOUNT + 1):
        group = {
            "id": group_id,
            "name": rands(),
            "label": rands(),
            "uid": uid_generator() if group_id > 4 else None,
            "parent_groups": []
        }
        if group_id > 2 and (randn(max=100) % 2 == 0):
            # Добавляем группу в группы
            group['parent_groups'].append(random.choice(groups)['id'])

        update_bb(org_id, group['uid'], f"{group['label']}@{domain}", is_maillist=True)
        groups.append(group)

    return groups


def _create_departments(org_id: int, domain: str, groups: list) -> Dict[int, dict]:
    departments: Dict[int, dict] = {}
    for department_id in range(1, settings.MOCK_DIRECTORY_DEPARTMENTS_AMOUNT + 1):
        with_parent = department_id > 2 and (randn(max=100) % 3 == 0)
        department = {
            "id": department_id,
            "name": rands() if department_id > 1 else 'Все отделы',
            "label": rands() if department_id > 1 else 'all',
            "uid": uid_generator(),
            "parent_id":
                randn(min=2, max=department_id - 1)
                if with_parent
                else (1 if department_id != 1 else None),
            "parent_groups": []
        }

        if department_id > 1 and (randn(max=100) % 2 == 0):
            parent_groups = set()
            for _ in range(randn(max=5)):
                parent_groups.add(random.choice(groups)['id'])  # type: ignore
            department['parent_groups'] = list(parent_groups)

        update_bb(org_id, department['uid'], f"{department['label']}@{domain}", is_maillist=True)
        departments[department_id] = department

    return departments


async def get_org(org_id: int, init_revision: int = 1) -> dict:
    async with locks[org_id]:
        if org_id in ORGS:
            return ORGS[org_id]

        domain = randdomain()
        name = rands()

        groups = _create_groups(org_id, domain)
        departments = _create_departments(org_id, domain, groups)

        users = []
        emails: Set[str] = set()
        for user_id in range(1, settings.MOCK_DIRECTORY_USERS_AMOUNT):
            while True:
                email = rands() + '@ya.ru'
                if email not in emails:
                    emails.add(email)
                    break

            user = {
                "id": uid_generator(),
                "name": {
                    "last": rands(),
                    "first": rands(),
                    "middle": rands()
                },
                "email": email,
                "groups": [],
                "departments": []
            }

            # Добавляем пользователя в группы
            for _ in range(randn(min=0, max=min(10, settings.MOCK_DIRECTORY_GROUP_AMOUNT))):
                while True:
                    group_id = random.choice(groups)['id']
                    if group_id not in user['groups']:
                        user['groups'].append(group_id)
                        break

            # Выбираем в какой отдел добавляем пользователя, также формируем путь до отдела
            department_id = random.choice(list(departments.keys()))
            user['departments'] = [department_id]
            while departments[department_id]['parent_id']:
                department_id = departments[department_id]['parent_id']
                user['departments'].insert(0, department_id)

            users.append(user)
            update_bb(org_id, user['id'], email)

        ORGS[org_id] = {
            "id": org_id,
            "name": {
                "ru": name,
                "en": name,
            },
            "label": name,
            "domains": {
                "all": [domain],
                "master": domain,
                "display": domain,
            },
            'revision': init_revision,
            'departments': list(departments.values()),
            'groups': groups,
            'users': users,
        }

    return ORGS[org_id]


async def delete_org(org_id: int) -> None:
    async with locks[org_id]:
        if org_id not in ORGS:
            return
        del ORGS[org_id]


def update_bb(org_id: int, uid: int, login: str, is_maillist: bool = False) -> None:
    record = {
        'org_id': org_id,
        'uid': uid,
        'is_maillist': is_maillist,
        'login': login
    }

    if uid:
        BB['by_uid'][uid] = record

    if login:
        BB['by_login'][login] = record


async def userinfo(uid: Optional[int] = None, login: Optional[str] = None) -> dict:
    assert not all((uid, login)) and any((uid, login)), 'userinfo: uid or login'

    try:
        if uid:
            bb_user = BB['by_uid'][uid]
        else:
            bb_user = BB['by_login'][login]
    except KeyError:
        raise CoreDataError('Not found', params={'uid': uid, 'login': login})

    return {
        'id': bb_user['uid'],
        'address-list': [
            {'address': bb_user['login'], 'default': True}
        ]
    }
