import math
import random

from aiohttp import web

from mail.beagle.beagle.api.exceptions import APIException
from mail.beagle.beagle.api.handlers.base import BaseHandler as BHandler
from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.mocks.directory.data import BB, get_org, userinfo
from mail.beagle.beagle.utils.helpers import without_none


class BaseHandler(BHandler):
    @property
    def org_id(self):
        org_id = self.request.match_info.get('org_id') or \
            self.request.headers.get('X-ORG-ID') or \
            self.request.query.get('org_id')
        try:
            return int(org_id)
        except (TypeError, ValueError):
            raise APIException(400, 'Invalid org_id')

    @property
    def group_id(self):
        try:
            return int(self.request.match_info.get('group_id', 20))
        except (TypeError, ValueError):
            raise APIException(400, 'Invalid group_id')

    @property
    def per_page(self):
        try:
            return int(self.request.query.get('per_page', 20))
        except (TypeError, ValueError):
            raise APIException(400, 'Invalid per_page')

    @property
    def page(self):
        try:
            return int(self.request.query.get('page', 1))
        except (TypeError, ValueError):
            raise APIException(400, 'Invalid page')


class AmmoHandler(BaseHandler):
    async def get(self):
        result = []
        req_template = (
            '%s %s HTTP/1.1\r\n'
            '%s\r\n'
            '\r\n'
        )
        ammo_template = (
            '%d %s\n'
            '%s'
        )

        logins = [login for login in BB['by_login'].keys() if BB['by_login'][login]['is_maillist']]

        if len(logins) > 0:
            for _ in range(self.per_page):
                login = random.choice(logins)

                headers = (
                    'host: load.beagle.mail.yandex.net\r\n'
                    'user-agent: tank\r\n'
                    'accept: */*\r\n'
                    'Connection: Close'
                )

                req = req_template % (
                    'GET',
                    f'/api/v1/recipients?email_to={login}&email_from=swatsup@yandex-team.ru',
                    headers
                )

                result.append(ammo_template % (len(req), 'simple', req))

        return web.Response(text=''.join(result))


# Blackbox
class BlackboxMockHandler(BaseHandler):
    async def get(self):
        login = self.request.query.get('login')
        uid = self.request.query.get('uid')

        assert self.request.query.get('method') == 'userinfo', 'only userinfo'
        assert self.request.query.get('format') == 'json', 'only json'
        assert not all((uid, login)) and any((login, uid)), 'uid or login'

        users = []

        if login:
            users.append(await userinfo(login=login))
        elif uid:
            uids = list(map(int, uid.split(',')))
            for uid in uids:
                users.append(await userinfo(uid=uid))

        return {'users': users}


# Directory

class DepartmentsHandler(BaseHandler):
    async def get(self):
        org = await get_org(self.org_id)
        offset = (self.page - 1) * self.per_page

        departments = org['departments'][offset:(offset + self.per_page)]
        pages = math.ceil(len(org['departments']) / self.per_page)

        return {
            'links': without_none({
                'next':
                    f'{settings.MOCK_DIRECTORY_URL}/v11/departments/?page={self.page + 1}'
                    if org['departments'][(offset + self.per_page):]
                    else None
            }),
            'page': self.page,
            'pages': pages,
            'per_page': self.per_page,
            'result': [
                {
                    "id": department['id'],
                    "name": department['name'],
                    "label": department['label'],
                    "uid": department['uid'],
                    "parent_id": department['parent_id']
                } for department in departments
            ],
            'total': len(org['departments'])
        }


class GroupsHandler(BaseHandler):
    async def get(self):
        org = await get_org(self.org_id)
        offset = (self.page - 1) * self.per_page

        groups = org['groups'][offset:(offset + self.per_page)]
        pages = math.ceil(len(org['groups']) / self.per_page)

        return {
            'links': without_none({
                'next':
                    f'{settings.MOCK_DIRECTORY_URL}/v11/groups/?page={self.page + 1}'
                    if org['groups'][(offset + self.per_page):]
                    else None
            }),
            'page': self.page,
            'pages': pages,
            'per_page': self.per_page,
            'result': [
                {
                    "id": group['id'],
                    "name": group['name'],
                    "label": group['label'],
                    "uid": group['uid']
                } for group in groups
            ],
            'total': len(org['groups'])
        }


class GroupsMembersHandler(BaseHandler):
    async def get(self):
        result = []
        org = await get_org(self.org_id)

        for user in org['users']:
            if self.group_id in user['groups']:
                result.append({
                    "type": "user",
                    "object": {
                        "department_id": user['departments'][-1],
                        "gender": "male",
                        "id": user['id'],
                        "name": user['name'],
                        "email": user['email'],
                        "position": None,
                        "avatar_id": "0/0-0"
                    }
                })

        for group in org['groups']:
            if self.group_id in group['parent_groups']:
                result.append({
                    "type": "group",
                    "object": {
                        "id": group['id'],
                        "members_count": None,
                        "name": group['name'],
                        "type": "generic"
                    }
                })

        for department in org['departments']:
            if self.group_id in department['parent_groups']:
                result.append({
                    "type": "department",
                    "object": {
                        "id": department['id'],
                        "members_count": None,
                        "name": department['name'],
                        "parent_id": department['parent_id']
                    }
                })

        return result


class OrganizationsHandler(BaseHandler):
    async def get(self):
        org = await get_org(self.org_id)

        return {
            'id': org['id'],
            'revision': org['revision'],
        }


class UsersHandler(BaseHandler):
    async def get(self):
        org = await get_org(self.org_id)
        offset = (self.page - 1) * self.per_page

        users = org['users'][offset:(offset + self.per_page)]
        pages = math.ceil(len(org['users']) / self.per_page)

        return {
            'links': without_none({
                'next':
                    f'{settings.MOCK_DIRECTORY_URL}/v11/users/?page={self.page + 1}'
                    if org['users'][(offset + self.per_page):]
                    else None
            }),
            'page': self.page,
            'pages': pages,
            'per_page': self.per_page,
            'result': [
                {
                    "id": user['id'],
                    "name": user['name'],
                    "email": user['email'],
                    "groups": [{'id': group_id} for group_id in user['groups']],
                    "departments": [{'id': department_id} for department_id in user['departments']]
                } for user in users
            ],
            'total': len(org['users'])
        }


class DomainsHandler(BaseHandler):
    async def get(self):
        org = await get_org(self.org_id)

        return [
            {
                "name": org['domains']['master'],
                "master": True
            }
        ]
