import sqlalchemy as sa
from sqlalchemy.dialects.postgresql import JSONB

from sendr_taskqueue.worker.storage.db.tables import get_task_state, get_task_type
from sendr_utils import enum_values

from mail.beagle.beagle.core.entities.enums import MailListType, SubscriptionType, TaskType

metadata = sa.MetaData(schema='beagle')

mail_list_type = sa.Enum(
    MailListType, name='mail_list_type', metadata=metadata, values_callable=enum_values
)
subscription_type = sa.Enum(
    SubscriptionType, name='subscription_type', metadata=metadata, values_callable=enum_values
)

organizations = sa.Table(
    'organizations', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('revision', sa.BigInteger(), default=0),
    sa.Column('is_deleted', sa.Boolean(), default=False, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

serials = sa.Table(
    'serials', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('mail_list_id', sa.BigInteger(), default=1),
    sa.Column('unit_id', sa.BigInteger(), default=1),
)

mail_lists = sa.Table(
    'mail_lists', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('mail_list_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('mail_list_type', mail_list_type, nullable=False),
    sa.Column('uid', sa.BigInteger(), nullable=False),
    sa.Column('username', sa.Text(), nullable=False),
    sa.Column('description', JSONB(), nullable=True),
    sa.Column('is_deleted', sa.Boolean(), default=False, nullable=False),
    sa.Column('settings', JSONB(), nullable=True),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

users = sa.Table(
    'users', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('username', sa.Text(), nullable=False),
    sa.Column('first_name', sa.Text(), nullable=False),
    sa.Column('last_name', sa.Text(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)
users_organizations_foreign_key = 'fk_users_on_organizations_org_id'

units = sa.Table(
    'units', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('unit_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), default=None, nullable=True),
    sa.Column('username', sa.Text(), nullable=True),
    sa.Column('external_id', sa.Text(), nullable=False),
    sa.Column('external_type', sa.Text(), nullable=False),
    sa.Column('name', sa.Text(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

unit_units = sa.Table(
    'unit_units', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('unit_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('parent_unit_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

unit_users = sa.Table(
    'unit_users', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('unit_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

unit_subscriptions = sa.Table(
    'unit_subscriptions', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('mail_list_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('unit_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('subscription_type', subscription_type, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

user_subscriptions = sa.Table(
    'user_subscriptions', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('mail_list_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('subscription_type', subscription_type, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

smtp_caches = sa.Table(
    'smtp_caches', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('value', JSONB(), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

mail_list_responsibles = sa.Table(
    'mail_list_responsibles', metadata,
    sa.Column('org_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('mail_list_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('uid', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), default=sa.func.now(), nullable=False),
)

tasks = sa.Table(
    'tasks', metadata,
    sa.Column('task_id', sa.BigInteger(), primary_key=True, nullable=False),
    sa.Column('task_type', get_task_type(metadata, TaskType), nullable=False),
    sa.Column('state', get_task_state(metadata), nullable=False),
    sa.Column('action_name', sa.Text()),
    sa.Column('org_id', sa.BigInteger(), nullable=True, default=None),
    sa.Column('params', JSONB()),
    sa.Column('details', JSONB()),
    sa.Column('retries', sa.Integer(), nullable=False, default=0),
    sa.Column('run_at', sa.DateTime(timezone=True), nullable=False),
    sa.Column('created', sa.DateTime(timezone=True), nullable=False),
    sa.Column('updated', sa.DateTime(timezone=True), nullable=False),
)
