from typing import AsyncIterable, Iterable, Optional

from sqlalchemy import collate, func

from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.beagle.beagle.core.entities.mail_list import MailList, MailListDescription
from mail.beagle.beagle.storage.db.tables import mail_lists as t_mail_lists
from mail.beagle.beagle.storage.exceptions import MailListNotFound, OrganizationNotFound, SerialNotFound
from mail.beagle.beagle.storage.mappers.base import BaseMapper
from mail.beagle.beagle.storage.mappers.serial_mixin import SerialMixin


class MailListDataMapper(SelectableDataMapper):
    entity_class = MailList
    selectable = t_mail_lists

    def map_description(self, data: dict) -> MailListDescription:
        data = data or {}
        return MailListDescription(**data)


class MailListDataDumper(TableDataDumper):
    entity_class = MailList
    table = t_mail_lists


class MailListMapper(SerialMixin, BaseMapper):
    name = 'mail_list'
    _builder = CRUDQueries(
        base=t_mail_lists,
        id_fields=('org_id', 'mail_list_id'),
        mapper_cls=MailListDataMapper,
        dumper_cls=MailListDataDumper,
    )

    async def create(self, mail_list: MailList) -> MailList:
        mail_list.created = mail_list.updated = func.now()
        try:
            mail_list.mail_list_id = await self.acquire_mail_list_id(mail_list.org_id)
        except SerialNotFound:
            raise OrganizationNotFound
        query, mapper = self._builder.insert(mail_list)
        return mapper(await self._query_one(query))

    async def delete(self, mail_list: MailList) -> MailList:
        mail_list.is_deleted = True
        mail_list.updated = func.now()
        query, mapper = self._builder.update(mail_list, only_fields=('is_deleted', 'updated'))
        return mapper(await self._query_one(query, raise_=MailListNotFound))

    async def get(self,
                  org_id: int,
                  mail_list_id: int,
                  is_deleted: Optional[bool] = False,
                  for_update: bool = False,
                  ) -> MailList:
        filters = Filters()
        filters.add_not_none('is_deleted', is_deleted)
        query, mapper = self._builder.select(
            id_values=(org_id, mail_list_id),
            filters=filters,
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=MailListNotFound))

    async def get_by_uid(self, uid: int) -> MailList:
        query, mapper = self._builder.select(filters={'uid': uid, 'is_deleted': False})
        return mapper(await self._query_one(query, raise_=MailListNotFound))

    async def find(self,
                   org_id: int,
                   mail_list_id: Optional[int] = None,
                   uids: Optional[Iterable[int]] = None,
                   is_deleted: Optional[bool] = False,
                   iterator: bool = False,
                   name_query: Optional[str] = None,
                   ) -> AsyncIterable[MailList]:
        filters = Filters()
        filters['org_id'] = org_id
        filters.add_not_none('mail_list_id', mail_list_id)
        filters.add_not_none('is_deleted', is_deleted)
        filters.add_not_none('uid', uids, lambda field: field.in_(uids))

        query, mapper = self._builder.select(filters=filters)
        if name_query is not None:
            name_query = name_query.lower()
            query = query.where(
                func.lower(collate(t_mail_lists.c.username, "C.UTF-8")).contains(name_query, autoescape=True)
            )
        async for row in self._query(query, iterator=iterator):
            yield mapper(row)

    async def save(self, mail_list: MailList) -> MailList:
        mail_list.updated = func.now()
        query, mapper = self._builder.update(mail_list, ignore_fields=('mail_list_id', 'created'))
        return mapper(await self._query_one(query, raise_=MailListNotFound))
