
from typing import AsyncIterable, Optional

import psycopg2
from sqlalchemy import func

from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.beagle.beagle.core.entities.mail_list_responsible import MailListResponsible
from mail.beagle.beagle.storage.db.tables import mail_list_responsibles as t_mail_list_responsibles
from mail.beagle.beagle.storage.exceptions import MailListResponsibleAlreadyExists, MailListResponsibleNotFound
from mail.beagle.beagle.storage.mappers.base import BaseMapper


class MailListResponsibleDataMapper(SelectableDataMapper):
    entity_class = MailListResponsible
    selectable = t_mail_list_responsibles


class MailListResponsibleDataDumper(TableDataDumper):
    entity_class = MailListResponsible
    table = t_mail_list_responsibles


class MailListResponsibleMapper(BaseMapper):
    name = 'mail_list_responsible'
    _builder = CRUDQueries(
        base=t_mail_list_responsibles,
        id_fields=('org_id', 'mail_list_id', 'uid'),
        mapper_cls=MailListResponsibleDataMapper,
        dumper_cls=MailListResponsibleDataDumper,
    )

    async def create(self, mail_list_responsible: MailListResponsible) -> MailListResponsible:
        mail_list_responsible.created = mail_list_responsible.updated = func.now()
        query, mapper = self._builder.insert(mail_list_responsible)
        try:
            return mapper(await self._query_one(query))
        except psycopg2.errors.UniqueViolation:
            raise MailListResponsibleAlreadyExists

    async def delete(self, mail_list_responsible: MailListResponsible) -> None:
        query = self._builder.delete(mail_list_responsible)
        await self._query_one(query)

    async def get(self,
                  org_id: int,
                  mail_list_id: int,
                  uid: int,
                  for_update: bool = False
                  ) -> MailListResponsible:
        query, mapper = self._builder.select(
            id_values=(org_id, mail_list_id, uid),
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=MailListResponsibleNotFound))

    async def find(self,
                   org_id: int,
                   mail_list_id: int,
                   uid: Optional[int] = None,
                   limit: Optional[int] = None,
                   offset: Optional[int] = None,
                   ) -> AsyncIterable[MailListResponsible]:
        filters = Filters()
        filters['org_id'] = org_id
        filters['mail_list_id'] = mail_list_id
        filters.add_not_none('uid', uid)
        query, mapper = self._builder.select(
            filters=filters,
            offset=offset,
            limit=limit,
        )
        async for row in self._query(query):
            yield mapper(row)
