from typing import Optional

import psycopg2
from sqlalchemy import func

from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import CRUDQueries, Filters

from mail.beagle.beagle.core.entities.organization import Organization
from mail.beagle.beagle.storage.db.tables import organizations as t_organizations
from mail.beagle.beagle.storage.exceptions import OrganizationAlreadyExists, OrganizationNotFound
from mail.beagle.beagle.storage.mappers.base import BaseMapper


class OrganizationDataMapper(SelectableDataMapper):
    entity_class = Organization
    selectable = t_organizations


class OrganizationDataDumper(TableDataDumper):
    entity_class = Organization
    table = t_organizations


class OrganizationMapper(BaseMapper):
    name = 'organization'
    _builder = CRUDQueries(
        base=t_organizations,
        id_fields=('org_id',),
        mapper_cls=OrganizationDataMapper,
        dumper_cls=OrganizationDataDumper,
    )

    async def create(self, organization: Organization) -> Organization:
        organization.created = organization.updated = func.now()
        query, mapper = self._builder.insert(organization)
        try:
            return mapper(await self._query_one(query))
        except psycopg2.errors.UniqueViolation:
            raise OrganizationAlreadyExists

    async def delete(self, organization: Organization) -> Organization:
        organization.is_deleted = True
        organization.updated = func.now()
        query, mapper = self._builder.update(organization, only_fields=('is_deleted', 'updated'))
        return mapper(await self._query_one(query, raise_=OrganizationNotFound))

    async def get(self,
                  org_id: int,
                  skip_deleted: Optional[bool] = True,
                  for_update: bool = False,
                  ) -> Organization:
        filters = Filters()
        if skip_deleted:
            filters.add_not_none('is_deleted', False)
        query, mapper = self._builder.select(
            id_values=(org_id,),
            filters=filters,
            for_update=for_update,
        )
        return mapper(await self._query_one(query, raise_=OrganizationNotFound))

    async def save(self, organization: Organization) -> Organization:
        organization.updated = func.now()
        query, mapper = self._builder.update(organization, ignore_fields=('created',))
        return mapper(await self._query_one(query, raise_=OrganizationNotFound))
