from typing import Optional

from sqlalchemy import func

from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import CRUDQueries, RelationDescription

from mail.beagle.beagle.core.entities.smtp_cache import CacheSubscription, CacheValue, SMTPCache
from mail.beagle.beagle.storage.db.tables import mail_lists as t_mail_lists
from mail.beagle.beagle.storage.db.tables import smtp_caches as t_smtp_caches
from mail.beagle.beagle.storage.exceptions import SMTPCacheNotFound
from mail.beagle.beagle.storage.mappers.base import BaseMapper
from mail.beagle.beagle.storage.mappers.mail_list import MailListDataMapper


class SMTPCacheDataMapper(SelectableDataMapper):
    entity_class = SMTPCache
    selectable = t_smtp_caches

    def map_value(self, value: dict) -> CacheValue:
        return CacheValue(subscriptions=[CacheSubscription(**subscr) for subscr in value['subscriptions']])


class SMTPCacheDataDumper(TableDataDumper):
    entity_class = SMTPCache
    table = t_smtp_caches

    @staticmethod
    def dump_value(value: CacheValue) -> Optional[dict]:
        if value:
            return {
                'subscriptions': [{'uid': subscr.uid,
                                   'local_part': subscr.local_part,
                                   'org_id': subscr.org_id,
                                   'params': subscr.params}
                                  for subscr in value.subscriptions
                                  ]
            }
        return None


class SMTPCacheMapper(BaseMapper):
    name = 'smtp_cache'
    _mail_list_relation = RelationDescription(
        name='mail_list',
        base=t_smtp_caches,
        related=t_mail_lists,
        mapper_cls=MailListDataMapper,
        base_cols=('org_id', 'uid'),
        related_cols=('org_id', 'uid'),
    )
    _builder = CRUDQueries(
        base=t_smtp_caches,
        id_fields=('org_id', 'uid'),
        mapper_cls=SMTPCacheDataMapper,
        dumper_cls=SMTPCacheDataDumper,
        related=(_mail_list_relation,),
    )

    def _map_related(row, mapper, rel_mappers):
        smtp_cache = mapper(row)
        smtp_cache.mail_list = rel_mappers['mail_list'](row)
        return smtp_cache

    async def create_or_update(self, smtp_cache: SMTPCache) -> SMTPCache:
        smtp_cache.created = smtp_cache.updated = func.now()
        query, mapper = self._builder.insert(smtp_cache, on_conflict_do_update_constraint=t_smtp_caches.primary_key)
        return mapper(await self._query_one(query))

    async def delete(self, smtp_cache: SMTPCache) -> None:
        query = self._builder.delete(smtp_cache)
        await self._query_one(query)

    async def get(self, org_id: int, uid: int) -> SMTPCache:
        query, mapper = self._builder.select(id_values=(org_id, uid))
        return mapper(await self._query_one(query, raise_=SMTPCacheNotFound))

    async def get_by_uid(self, uid: int) -> SMTPCache:
        query, mapper, rel_mappers = self._builder.select_related(filters={'uid': uid, 'mail_list.is_deleted': False})
        return mapper(await self._query_one(query, raise_=SMTPCacheNotFound))

    async def save(self, smtp_cache: SMTPCache) -> SMTPCache:
        smtp_cache.updated = func.now()
        query, mapper = self._builder.update(smtp_cache, ignore_fields=('created',))
        return mapper(await self._query_one(query, raise_=SMTPCacheNotFound))
