from typing import AsyncIterable

from sqlalchemy import func

from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import CRUDQueries

from mail.beagle.beagle.core.entities.unit import Unit
from mail.beagle.beagle.storage.db.tables import units as t_units
from mail.beagle.beagle.storage.exceptions import UnitNotFound
from mail.beagle.beagle.storage.mappers.base import BaseMapper
from mail.beagle.beagle.storage.mappers.serial_mixin import SerialMixin


class UnitDataMapper(SelectableDataMapper):
    entity_class = Unit
    selectable = t_units


class UnitDataDumper(TableDataDumper):
    entity_class = Unit
    table = t_units


class UnitMapper(SerialMixin, BaseMapper):
    name = 'unit'
    _builder = CRUDQueries(
        base=t_units,
        id_fields=('org_id', 'unit_id'),
        mapper_cls=UnitDataMapper,
        dumper_cls=UnitDataDumper,
    )

    async def create(self, unit: Unit) -> Unit:
        unit.unit_id = await self.acquire_unit_id(unit.org_id)
        unit.created = unit.updated = func.now()
        query, mapper = self._builder.insert(unit)
        return mapper(await self._query_one(query))

    async def delete(self, unit: Unit) -> None:
        query = self._builder.delete(unit)
        await self._query_one(query)

    async def find(self, org_id: int) -> AsyncIterable[Unit]:
        query, mapper = self._builder.select(id_values=(org_id,))
        async for row in self._query(query):
            yield mapper(row)

    async def get(self, org_id: int, unit_id: int) -> Unit:
        query, mapper = self._builder.select(id_values=(org_id, unit_id))
        return mapper(await self._query_one(query, raise_=UnitNotFound))

    async def save(self, unit: Unit) -> Unit:
        unit.updated = func.now()
        query, mapper = self._builder.update(unit, ignore_fields=('created',))
        return mapper(await self._query_one(query, raise_=UnitNotFound))
