from typing import Any, AsyncIterable, Dict, Optional

from sqlalchemy import func

from sendr_aiopg.data_mapper import SelectableDataMapper, TableDataDumper
from sendr_aiopg.query_builder import CRUDQueries, Filters, RelationDescription

from mail.beagle.beagle.core.entities.unit_user import UnitUser
from mail.beagle.beagle.storage.db.tables import unit_subscriptions as t_unit_subscriptions
from mail.beagle.beagle.storage.db.tables import unit_users as t_unit_users
from mail.beagle.beagle.storage.exceptions import UnitUserNotFound
from mail.beagle.beagle.storage.mappers.base import BaseMapper
from mail.beagle.beagle.storage.mappers.unit_subscription import UnitSubscriptionDataMapper


class UnitUserDataMapper(SelectableDataMapper):
    entity_class = UnitUser
    selectable = t_unit_users


class UnitUserDataDumper(TableDataDumper):
    entity_class = UnitUser
    table = t_unit_users


class UnitUserMapper(BaseMapper):
    name = 'unit_user'
    _unit_subscription_relation = RelationDescription(
        name='unit_subscription',
        base=t_unit_users,
        related=t_unit_subscriptions,
        mapper_cls=UnitSubscriptionDataMapper,
        base_cols=('org_id', 'unit_id'),
        related_cols=('org_id', 'unit_id'),
    )
    _builder = CRUDQueries(
        base=t_unit_users,
        id_fields=('org_id', 'unit_id', 'uid'),
        mapper_cls=UnitUserDataMapper,
        dumper_cls=UnitUserDataDumper,
        related=(_unit_subscription_relation,)
    )

    async def create(self, unit_user: UnitUser) -> UnitUser:
        unit_user.created = unit_user.updated = func.now()
        query, mapper = self._builder.insert(unit_user)
        return mapper(await self._query_one(query))

    async def delete(self, unit_user: UnitUser) -> None:
        query = self._builder.delete(unit_user)
        await self._query_one(query)

    async def find(self,
                   org_id: int,
                   uid: Optional[int] = None,
                   unit_subscription_mail_list_id: Optional[int] = None,
                   order_by: Optional[str] = None,
                   iterator: bool = False
                   ) -> AsyncIterable[UnitUser]:
        filters = Filters()
        filters.add_not_none('uid', uid)
        filters.add_not_none('unit_subscription.mail_list_id', unit_subscription_mail_list_id)

        kwargs: Dict[str, Any] = {
            'id_values': (org_id,),
            'filters': filters,
            'order': (order_by,) if order_by else None
        }

        if unit_subscription_mail_list_id:
            query, mapper, _ = self._builder.select_related(**kwargs)
        else:
            query, mapper = self._builder.select(**kwargs)

        async for row in self._query(query, iterator=iterator):
            yield mapper(row)

    async def get(self, org_id: int, unit_id: int, uid: int) -> UnitUser:
        query, mapper = self._builder.select(id_values=(org_id, unit_id, uid))
        return mapper(await self._query_one(query, raise_=UnitUserNotFound))

    async def save(self, unit_user: UnitUser) -> UnitUser:
        unit_user.updated = func.now()
        query, mapper = self._builder.update(unit_user, ignore_fields=('created',))
        return mapper(await self._query_one(query, raise_=UnitUserNotFound))
