import asyncio
import logging
import uuid

import uvloop

from sendr_qlog import LoggerContext

from mail.beagle.beagle.conf import settings
from mail.beagle.beagle.core.actions.base import BaseAction
from mail.beagle.beagle.interactions import InteractionClients
from mail.beagle.beagle.interactions.base import BaseInteractionClient, create_connector
from mail.beagle.beagle.storage import Storage
from mail.beagle.beagle.utils.db import create_configured_engine
from mail.beagle.beagle.utils.logging import configure_logging


def create_shell_context():
    asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())
    loop = asyncio.get_event_loop()

    configure_logging()

    BaseInteractionClient.CONNECTOR = create_connector()
    request_id = 'manual_' + uuid.uuid4().hex
    logger = LoggerContext(logging.getLogger(), {'request_id': request_id})
    clients = InteractionClients(logger, request_id)

    db_engine = create_configured_engine()

    conn = loop.run_until_complete(db_engine.acquire().__aenter__())
    storage = Storage(conn)

    BaseAction.context.logger = logger
    BaseAction.context.request_id = request_id
    BaseAction.context.db_engine = db_engine

    return {
        'clients': clients,
        'db_engine': db_engine,
        'logger': logger,
        'request_id': request_id,
        'settings': settings,
        'storage': storage,
    }


def action_command(func):
    """
    Example:
    ```
    @click.command()
    @click.argument('service_id', type=int)
    @action_command
    async def cli(context, service_id):
        context['service_id'] = service_id
        await SomeAction(context).run()
    ```
    """

    def _inner(**kwargs):
        create_shell_context()
        loop = asyncio.get_event_loop()

        try:
            return loop.run_until_complete(func(**kwargs))
        finally:
            loop.run_until_complete(BaseInteractionClient.close_connector())

    return _inner
