from enum import Enum, unique
from typing import Dict, Hashable, List, Set, TypeVar

NodeType = TypeVar('NodeType', bound=Hashable)
GraphType = Dict[NodeType, Set[NodeType]]


class GraphError(Exception):
    pass


class CycleFoundError(GraphError):
    pass


@unique
class NodeState(Enum):
    READY = 0
    IN = 1
    OUT = 2


def expand_acyclic_graph(graph: GraphType[NodeType]) -> GraphType:
    """
    Returns graph where each child (not only direct) of each node is present in graph[node].
    """
    expanded: GraphType[NodeType] = {}
    node_states: Dict[NodeType, NodeState] = {}

    for starting_node in graph:
        if starting_node in node_states:
            continue
        node_states[starting_node] = NodeState.READY
        stack: List[NodeType] = [starting_node]
        while stack:
            node = stack.pop()
            state = node_states[node]
            children = graph.get(node, set())
            if state == NodeState.READY:
                node_states[node] = NodeState.IN
                stack.append(node)
                for child in children:
                    if child not in node_states:
                        node_states[child] = NodeState.READY
                        stack.append(child)
                    elif node_states[child] == NodeState.IN:
                        raise CycleFoundError
            elif state == NodeState.IN:
                expanded[node] = set.union(
                    {child for child in children},
                    *[expanded[child] for child in children],
                )
                node_states[node] = NodeState.OUT
            else:
                raise GraphError('Unexpected node state.')

    return expanded
