#pragma once

#include <yplatform/encoding/base64.h>
#include <openssl/evp.h>
#include <cassert>
#include <fstream>
#include <map>
#include <sstream>
#include <stdexcept>
#include <vector>

namespace blowfish {

using std::string;

namespace detail {

template <class Range>
inline string make_string(const Range& r)
{
    return string(r.begin(), r.end());
}

inline string pad(const string& in, size_t size, char ch)
{
    string padded(size, ch);
    std::copy(in.begin(), in.begin() + std::min(size, in.size()), padded.begin());
    return padded;
}

inline string pbkdf2(const string& key)
{
    char result[16];
    const char* salt = "mEcvNvLr";
    if (0 ==
        PKCS5_PBKDF2_HMAC(
            key.c_str(),
            static_cast<int>(key.size()),
            reinterpret_cast<const unsigned char*>(salt),
            8,
            1000,
            EVP_md5(),
            16,
            reinterpret_cast<unsigned char*>(result)))
        throw std::runtime_error("PKCS5_PBKDF2_HMAC");
    return string(result, result + 16);
}
} // detail

inline string encrypt(const string& in, const string& key, const string& iv)
{
    assert(key.size() == 16);
    assert(iv.size() == 8);

    struct guard
    {
        EVP_CIPHER_CTX* ctx;
        guard() : ctx(EVP_CIPHER_CTX_new())
        {
            if (ctx == nullptr)
            {
                throw std::runtime_error("Failed to allocate EVP_CIPHER_CTX");
            }
        }
        ~guard()
        {
            EVP_CIPHER_CTX_free(ctx);
        }

    } ctx_guard;
    EVP_CIPHER_CTX* ctx = ctx_guard.ctx;

    EVP_EncryptInit_ex(
        ctx,
        EVP_bf_cbc(),
        NULL,
        reinterpret_cast<const unsigned char*>(key.c_str()),
        reinterpret_cast<const unsigned char*>(iv.c_str()));
    std::vector<unsigned char> out(in.size() + EVP_CIPHER_block_size(EVP_bf_cbc()) + 1024);

    int outlen = 0, tmplen = 0;
    if (!EVP_EncryptUpdate(
            ctx,
            out.data(),
            &outlen,
            reinterpret_cast<const unsigned char*>(in.data()),
            static_cast<int>(in.size())))
        throw std::runtime_error("EVP_EncryptUpdate error");
    if (!EVP_EncryptFinal_ex(ctx, out.data() + outlen, &tmplen))
        throw std::runtime_error("EVP_EncryptFinal_ex error");

    outlen += tmplen;
    return string(out.begin(), out.begin() + outlen);
}

inline string decrypt(const string& in, const string& key, const string& iv)
{
    assert(key.size() == 16);
    assert(iv.size() == 8);

    std::vector<unsigned char> out(in.size() + 1024);
    int outlen = 0, tmplen = 0;

    struct guard
    {
        EVP_CIPHER_CTX* ctx;
        guard() : ctx(EVP_CIPHER_CTX_new())
        {
            if (ctx == nullptr)
            {
                throw std::runtime_error("Failed to allocate EVP_CIPHER_CTX");
            }
        }
        ~guard()
        {
            EVP_CIPHER_CTX_free(ctx);
        }
    } ctx_guard;
    EVP_CIPHER_CTX* ctx = ctx_guard.ctx;

    EVP_DecryptInit_ex(
        ctx,
        EVP_bf_cbc(),
        NULL,
        reinterpret_cast<const unsigned char*>(key.c_str()),
        reinterpret_cast<const unsigned char*>(iv.c_str()));
    if (!EVP_DecryptUpdate(
            ctx,
            out.data(),
            &outlen,
            reinterpret_cast<const unsigned char*>(in.data()),
            static_cast<int>(in.size())))
        throw std::runtime_error("EVP_DecryptUpdate");
    if (!EVP_DecryptFinal_ex(ctx, out.data() + outlen, &tmplen))
        throw std::runtime_error("EVP_DecryptFinal_ex");
    outlen += tmplen;

    string res = string(out.begin(), out.begin() + outlen);
    return res;
}

namespace password {

using versioned_keys = std::map<uint32_t, string>;

inline versioned_keys compute_derived_keys(const string& keys_filename)
{
    // Format of the keys file: [timestamp key]
    std::ifstream keys(keys_filename.c_str());
    if (!keys.is_open())
        throw std::runtime_error(string("failed to open keys config ").append(keys_filename));

    versioned_keys dkeys;
    while (keys)
    {
        uint32_t timestamp;
        string key;
        if (keys >> timestamp)
        {
            if (!(keys >> key))
                throw std::runtime_error(
                    string("failed to parse keys config ").append(keys_filename));

            string dkey = detail::pbkdf2(key);
            dkeys.insert(std::make_pair(timestamp, dkey));
        }
    }

    if (dkeys.empty())
    {
        throw std::runtime_error(
            string("failed to parse keys config (or empty) ").append(keys_filename));
    }

    return dkeys;
}

inline string encrypt(const string& in, const versioned_keys& dkeys, const string& iv)
{
    try
    {
        if (dkeys.empty()) throw std::runtime_error("no keys");

        enum
        {
            PASSWORD_IS_ENCRYPTED = 9
        };

        auto it = dkeys.rbegin();

        uint32_t network_order_timestamp = htonl(it->first);
        char timestamp[sizeof(network_order_timestamp)];
        memcpy(timestamp, &network_order_timestamp, sizeof(network_order_timestamp));
        string key = it->second;

        string r = string(timestamp, timestamp + 4) +
            blowfish::encrypt(in, detail::pad(key, 16, ' '), detail::pad(iv, 8, ' '));
        return string(2, PASSWORD_IS_ENCRYPTED) +
            detail::make_string(yplatform::base64_encode(r.begin(), r.end()));
    }
    catch (const std::exception& e)
    {
        throw std::runtime_error(
            string("failed to encrypt password (").append(e.what()).append("): ").append(in));
    }
}

inline string decrypt(const string& in64, const versioned_keys& dkeys, const string& iv)
{
    try
    {
        string in = detail::make_string(yplatform::base64_decode(in64.begin() + 2, in64.end()));

        if (in.size() < 4) throw std::runtime_error("password too short");

        uint32_t timestamp;
        memcpy(&timestamp, in.c_str(), sizeof(timestamp));
        timestamp = ntohl(timestamp);

        auto it = dkeys.find(timestamp);
        if (it == dkeys.end())
        {
            std::ostringstream os;
            os << "failed to decrypt unknown password version " << timestamp;
            throw std::runtime_error(os.str());
        }

        return blowfish::decrypt(
            string(in.begin() + 4, in.end()),
            detail::pad(it->second, 16, ' '),
            detail::pad(iv, 8, ' '));
    }
    catch (const std::exception& e)
    {
        throw std::runtime_error(
            string("failed to decrypt password (").append(e.what()).append("): ").append(in64));
    }
}

} // password

} // blowfish
