import os
from contextlib import contextmanager

import requests
import yaml

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import SubprocessComponent, WithPort, BaseComponent
from library.python import resource
from mail.webmail_config.lib.make_config import make_config


class AquaCli(BaseComponent):
    NAME = 'aqua'
    DEPS = []

    def __init__(self, env, components):
        super().__init__(env, components)
        self.mock_path = os.path.join(self.root, 'mock')
        self.aqua_path = os.path.join(self.root, 'aqua')

    def get_etc_path(self):
        return self.root

    def get_log_dir(self):
        return self.root

    def init_root(self):
        mock = self.env.get_arcadia_bin('mail/borador/aqua/mock/aqua_cli')
        helpers.run_subprocess('cp', self.logger, ['cp', mock, f'{self.mock_path}'], env=None)
        helpers.run_subprocess('chmod', self.logger, ['chmod', '+x', f'{self.mock_path}'], env=None)

        jar = self.env.get_arcadia_bin('mail/borador/aqua/cli/aqua_cli.jar')
        helpers.run_subprocess('cp', self.logger, ['cp', jar, f'{self.aqua_path}'], env=None)

    def set_mock_port(self, port: int):
        helpers.write2file(path=os.path.join(self.root, 'aqua.url'), what=f'http://localhost:{port}/run')

    @contextmanager
    def starting(self, cmd, env=None):
        self.logger.info('Nothing to do for %s, info: %r', self.name, self.info())

    def stop(self):
        self.logger.info('Nothing to do for %s, info: %r', self.name, self.info())

    def info(self):
        base = super(BaseComponent, self).info()
        base.update({
            'aqua_path': self.aqua_path,
            'mock_path': self.mock_path,
        })
        return base


class BoradorTesting(SubprocessComponent, WithPort):
    NAME = 'borador_testing'
    DEPS = [AquaCli]

    def get_log_dir(self):
        return os.path.join(self.root, 'logs')

    def tskv_log(self):
        return os.path.join(self.get_log_dir(), 'borador.tskv')

    def __init__(self, env, components):
        super().__init__(env, components)
        self.bin_file = env.get_arcadia_bin('mail/borador/bin/borador')
        self.config_path = os.path.join(self.root, 'borador.yaml')
        self.aqua_path = os.path.join(self.root, 'aqua.xml')
        self.aqua_cli = self.components[AquaCli]

    def path_to_aqua(self):
        try:
            j = self.env.get_java_path().decode('utf-8')
        except:
            j = 'java'

        return [j, '-jar', self.aqua_cli.aqua_path]

    def wait_time(self):
        return 1

    def init_root(self):
        helpers.mkdir_recursive(self.log_dir)

        borador = yaml.full_load(resource.find('borador/borador.yaml'))
        service = yaml.safe_load(resource.find('borador/service.yaml'))
        aqua = resource.find('borador/aqua.xml')

        service['testing']['port'] = self.port
        service['testing']['path_to_aqua'] = self.path_to_aqua()

        borador['config']['level'] = 'debug'
        borador['config']['port'] = self.port
        borador['config']['log'] = self.tskv_log()
        borador['config']['launch']['path_to_aqua'] = self.path_to_aqua()
        borador['config']['launch']['additional_arguments'] = []
        if self.wait_time():
            borador['config']['launch']['wait_time'] = self.wait_time()

        helpers.write2file(yaml.dump(borador), self.config_path)
        helpers.write2file(make_config('testing', aqua, service, silent=True), self.aqua_path)

    def start(self):
        cmd = [self.bin_file, self.config_path]

        with self.starting(cmd, {}):
            helpers.wait_ping(self.logger, self.healthcheck)

    def start_pack(self, pack_name: str, x_request_id: str, timeout=2):
        url = f'http://localhost:{self.port}/launch?pack_name={pack_name}&x_request_id={x_request_id}'
        self.logger.info('POST request, url: %s', url)
        return requests.post(url, timeout=timeout)

    def request(self, args: str, timeout=2):
        url = f'http://localhost:{self.port}/launch' + args
        self.logger.info('POST request, url: %s', url)
        return requests.post(url, timeout=timeout)

    def healthcheck(self, timeout=1):
        url = f'http://localhost:{self.port}/ping'
        self.logger.info('GET request, url: %s', url)
        return requests.get(url, timeout=timeout)

    def info(self):
        return {
            'bin_file': self.bin_file,
            **super().info()
        }


class Borador(BoradorTesting):
    NAME = 'borador'
    DEPS = [AquaCli]

    def __init__(self, env, components):
        super().__init__(env, components)
        self.aqua_cli.set_mock_port(port=self.mock_port)

    @classmethod
    def gen_config(cls, port_generator, config=None):
        base = super(WithPort, cls).gen_config(port_generator, config=config)
        return dict(
            port=next(port_generator),
            mock_port=next(port_generator),
            **base
        )

    def wait_time(self):
        return None

    @property
    def mock_port(self):
        return self.config[self.name]['mock_port']

    def path_to_aqua(self):
        return [self.aqua_cli.mock_path]
