import os

import requests
import yaml

from mail.devpack.lib import helpers
from mail.devpack.lib.components.base import SubprocessComponent, WithPort
from mail.borador.devpack.components.borador import Borador, BoradorTesting
from library.python import resource


class Unistat(SubprocessComponent, WithPort):
    NAME = 'unistat'
    DEPS = [Borador]

    def get_log_dir(self):
        return os.path.join(self.root)

    def __init__(self, env, components):
        super().__init__(env, components)
        self.bin_file = env.get_arcadia_bin('mail/borador/unistat/cunistat')
        self.unistat_file = os.path.join(self.root, 'unistat.py')
        self.unistat_conf = os.path.join(self.root, 'unistat.yaml')

    def init_root(self):
        helpers.mkdir_recursive(self.log_dir)
        helpers.write2file(resource.find('borador/unistat.py'), self.unistat_file)

        config = yaml.full_load(resource.find('borador/unistat.yaml'))

        config['config']['host'] = 'localhost'
        config['config']['port'] = self.port
        config['config']['log'] = os.path.join(self.log_dir, 'unistat.log')
        config['config']['source'] = self.tskv_log()

        helpers.write2file(yaml.dump(config), self.unistat_conf)

    def tskv_log(self):
        return self.components[Borador].tskv_log()

    def start(self):
        cmd = [
            self.bin_file,
            self.unistat_file,
            '--config', self.unistat_conf
        ]

        with self.starting(cmd, {}):
            helpers.wait_ping(self.logger, self.healthcheck)

    def healthcheck(self, timeout=1):
        url = f'http://localhost:{self.port}/ping'
        self.logger.info('GET request, url: %s', url)
        return requests.get(url, timeout=timeout)

    def unistat(self, timeout=1):
        url = f'http://localhost:{self.port}/unistat'
        self.logger.info('GET request, url: %s', url)
        return requests.get(url, timeout=timeout)

    def info(self):
        return {
            'bin_file': self.bin_file,
            **super().info()
        }


class UnistatTesting(Unistat):
    NAME = 'unistat_testing'
    DEPS = [BoradorTesting]

    def __init__(self, env, components):
        super().__init__(env, components)

    def tskv_log(self):
        return self.components[BoradorTesting].tskv_log()
