import asyncio
import xml.etree.ElementTree
from dataclasses import dataclass
from typing import List


@dataclass
class LaunchId:
    id: str


@dataclass
class Show:
    launchId: str = ''
    totalSuites: int = 0
    passedSuites: int = 0
    launchStatus: str = ''
    startTime: int = 0
    stopTime: int = 0


def log_show(show: Show) -> List:
    return [('empty_show', False)] + list(show.__dict__.items()) if show is not None else [('empty_show', True)]


def parse_launch_pack(launch: str) -> LaunchId:
    root = xml.etree.ElementTree.fromstring(launch)

    xmlns = 'urn:beans.aqua.yandex.ru'
    for l_id in root.findall(f'.//{{{xmlns}}}launches/{{{xmlns}}}id'):
        return LaunchId(id=l_id.text)


def parse_restart_pack(launch: str) -> LaunchId:
    root = xml.etree.ElementTree.fromstring(launch)
    xmlns = 'urn:beans.aqua.yandex.ru'

    return LaunchId(id=root.findall(f'.{{{xmlns}}}id')[0].text)


def parse_launch_show(launch: str) -> Show:
    root = xml.etree.ElementTree.fromstring(launch)
    xmlns = 'urn:beans.aqua.yandex.ru'

    return Show(totalSuites=int(root.findall(f'.{{{xmlns}}}totalSuites')[0].text),
                passedSuites=int(root.findall(f'.{{{xmlns}}}passedSuites')[0].text),
                launchId=root.findall(f'.{{{xmlns}}}id')[0].text,
                launchStatus=root.findall(f'.{{{xmlns}}}launchStatus')[0].text,
                startTime=int(root.findall(f'.{{{xmlns}}}startTime')[0].text),
                stopTime=int(root.findall(f'.{{{xmlns}}}stopTime')[0].text))


class AquaApi(object):
    def __init__(self, logger, path_to_aqua: List[str], additional_arguments: List[str], wait_time: int, tag: str):
        self._path_to_aqua = path_to_aqua
        self._additional_arguments = additional_arguments
        self.wait_time = wait_time
        self.logger = logger
        self.pack_launched = asyncio.get_running_loop().create_future()
        self.tag = tag

    async def wait_for_pack_to_be_launched(self):
        await self.pack_launched

    async def _launch(self, cmd: List[str]) -> str:
        p = await asyncio.create_subprocess_exec(cmd[0], *cmd[1:], stderr=asyncio.subprocess.PIPE,
                                                 stdout=asyncio.subprocess.PIPE)
        out, err = await p.communicate()

        self.logger.info(('message', f'command "{cmd}" finished with status {p.returncode}'))

        if p.returncode != 0:
            self.logger.error(('stdout', str(out)), ('stderr', str(err)), ('retcode', p.returncode))
            raise Exception(str(err))

        self.logger.debug(('out', str(out)), ('err', str(err)))

        return out.decode('utf-8')

    async def launch_pack(self, pack_name: str) -> LaunchId:
        try:
            res = parse_launch_pack(launch=await self._launch(cmd=self._path_to_aqua + ['launch', 'pack'] +
                                                                  self._additional_arguments +
                                                                  ['--tag', self.tag, pack_name]))
        except Exception as e:
            self.pack_launched.set_exception(Exception('cannot launch pack'))
            raise e
        self.pack_launched.set_result(None)
        return res

    async def restart_launch(self, launch_id: str) -> LaunchId:
        return parse_restart_pack(launch=await self._launch(cmd=self._path_to_aqua + ['restart', '--failedOnly'] +
                                                                self._additional_arguments + [launch_id]))

    async def show_launch(self, launch_id: LaunchId) -> Show:
        return parse_launch_show(launch=await self._launch(cmd=self._path_to_aqua + ['show', 'launch'] +
                                                               self._additional_arguments + [launch_id.id]))

    async def wait_for_next_show(self):
        await asyncio.sleep(self.wait_time)


def pack_is_running(show: Show) -> bool:
    return show.stopTime == 0


def pack_is_success(show: Show) -> bool:
    return show is not None and show.totalSuites == show.passedSuites
