import asyncio
import json
import logging
import sys
from typing import Callable, Awaitable

from aiohttp import web

from mail.borador.src.aqua import AquaApi
from mail.borador.src.config import instance, load_config, get_environment
from mail.borador.src.is_master import process_is_master
from mail.borador.src.tasks import launch_pack_and_wait_for_finishing
from mail.borador.src.log import logger_init
from mail.borador.src import log


def with_access_log(func: Callable[[web.Request], Awaitable[web.Response]]):
    async def wrapper(request: web.Request) -> web.Response:
        with log.Flush():
            response = None
            try:
                response = await func(request)
            except Exception:
                raise
            finally:
                req_id = request.query['x_request_id'] if 'x_request_id' in request.query else '-'
                log.logger(request_id=req_id).info(
                    ('action', 'access'),
                    ('url', request.path),
                    ('status', response.status if hasattr(response, 'status') else '-'),
                )
            return response

    return wrapper


@with_access_log
async def ping(_: web.Request) -> web.Response:
    return web.Response(text='pong\n')


@with_access_log
async def launch(req: web.Request) -> web.Response:
    try:
        pack_name = req.query['pack_name']
        x_request_id = req.query['x_request_id']

        if not await process_is_master():
            return web.Response(text=json.dumps({'status': 'is not master'}),
                                headers={'Content-Type': 'application/json'}, status=200)

        cfg = instance().launch
        api = AquaApi(path_to_aqua=cfg.path_to_aqua,
                      additional_arguments=cfg.additional_arguments,
                      wait_time=cfg.wait_time, tag='borador_' + get_environment(),
                      logger=log.logger(request_id=x_request_id))

        asyncio.create_task(
            launch_pack_and_wait_for_finishing(api=api, pack_name=pack_name, config=instance().launch,
                                               x_request_id=x_request_id)
        )

        await api.wait_for_pack_to_be_launched()

        return web.Response(text=json.dumps({'status': 'launched'}),
                            headers={'Content-Type': 'application/json'}, status=200)
    except KeyError as e:
        return web.Response(text=json.dumps({'status': str(e)}),
                            headers={'Content-Type': 'application/json'}, status=400)
    except Exception as e:
        return web.Response(text=json.dumps({'status': str(e)}),
                            headers={'Content-Type': 'application/json'}, status=500)


def main():
    cfg = load_config(sys.argv[1])
    logger_init(log_file=cfg.log, level=cfg.level)

    app = web.Application()
    logging.basicConfig(level=logging.DEBUG)
    app.router.add_get('/ping', ping)
    app.router.add_post('/launch', launch)

    log.logger(request_id='global').info(('action', 'init'))

    web.run_app(app, host='::', port=instance().port,
                access_log_format='%a %t "%r" %s %b "%{Referer}i" "%{User-Agent}i" %Tf')


if __name__ == '__main__':
    main()
