import datetime
import logging
from logging.handlers import TimedRotatingFileHandler

from dateutil.tz import tzlocal
from mail.borador.src import tskv


_logger: logging.Logger = logging.getLogger('borador')


class Flush(object):
    def __enter__(self):
        return self

    def __exit__(self, t, value, traceback):
        for h in _logger.handlers:
            h.flush()
        return False


class Logger(object):
    def __init__(self, request_id):
        self.request_id = request_id

    def error(self, *args):
        _logger.error(msg='', extra=[('x_request_id', self.request_id), *args])

    def debug(self, *args):
        _logger.debug(msg='', extra=[('x_request_id', self.request_id), *args])

    def info(self, *args):
        _logger.info(msg='', extra=[('x_request_id', self.request_id), *args])


def logger(request_id):
    return Logger(request_id=request_id)


class ExtraTskvFormatter(logging.Formatter):
    TZ = tzlocal()

    def format(self, record):
        message = super(ExtraTskvFormatter, self).format(record)

        ts = datetime.datetime.fromtimestamp(record.created, tz=ExtraTskvFormatter.TZ)

        msg = [
            ('tskv_format', 'mail-barador-log'),
            ('timestamp', ts.strftime('%Y-%m-%d %H:%M:%S.%f')),
            ('unixtime', int(ts.timestamp())),
            ('level', record.levelname.lower()),
        ] + record.extra + ([('message', message)] if message is not None and message != '' else [])

        return tskv.format(values=msg)


def _monkey_patch_logger():
    def makeRecord(self, name, level, fn, lno, msg, args, exc_info,
                   func=None, extra=None, sinfo=None):
        rv = self.__makeRecord(
            name, level, fn, lno, msg, args, exc_info, func=func, extra=None, sinfo=sinfo
        )
        rv.__dict__['extra'] = extra
        return rv

    logging.getLoggerClass().__makeRecord = logging.getLoggerClass().makeRecord
    logging.getLoggerClass().makeRecord = makeRecord


def logger_init(log_file: str, level):
    global _logger

    _monkey_patch_logger()

    logging.addLevelName(logging.DEBUG, 'debug')
    logging.addLevelName(logging.INFO, 'info')
    logging.addLevelName(logging.WARNING, 'warning')
    logging.addLevelName(logging.ERROR, 'error')

    _logger = logging.Logger('borador')

    h = TimedRotatingFileHandler(filename=log_file, encoding='utf-8', backupCount=5, when='MIDNIGHT')
    h.setLevel(level)
    h.setFormatter(ExtraTskvFormatter())

    _logger.addHandler(h)
