import traceback
from typing import Tuple

from mail.borador.src.aqua import AquaApi, Show, pack_is_running, pack_is_success, log_show
from mail.borador.src.config import LaunchAquaPackConfig
from mail.borador.src.log import logger, Flush


async def launch_and_wait(api: AquaApi, failed_only: bool, **kwargs) -> Show:
    if failed_only:
        launch_id = await api.restart_launch(**kwargs)
    else:
        launch_id = await api.launch_pack(**kwargs)

    while True:
        show = await api.show_launch(launch_id=launch_id)

        if not pack_is_running(show):
            return show
        else:
            await api.wait_for_next_show()


async def with_retry(api: AquaApi, pack_name: str, times: int, x_request_id: str = '') -> Tuple[int, int, int]:
    failed_only = False
    show = None
    total: int = 0
    for i in range(times):
        if i == 0:
            show = await launch_and_wait(api=api, pack_name=pack_name, failed_only=False)
            total = show.totalSuites
        else:
            show = await launch_and_wait(api=api, launch_id=show.launchId, failed_only=True)

        logger(request_id=x_request_id).info(
            ('action', 'response_is_got'),
            ('pack_name', pack_name),
            ('n', str(i)),
            ('failed_only', str(failed_only)),
            *log_show(show)
        )

        if pack_is_success(show):
            return total, (show.totalSuites - show.passedSuites), i+1

    return total, (show.totalSuites - show.passedSuites), -1


async def launch_pack_and_wait_for_finishing(api: AquaApi, pack_name: str, x_request_id: str, config: LaunchAquaPackConfig):
    with Flush():
        try:
            logger(request_id=x_request_id).info(
                ('action', 'starting'),
                ('pack_name', pack_name),
            )

            total, failed, times = await with_retry(api=api, pack_name=pack_name, times=config.tries, x_request_id=x_request_id)

            logger(request_id=x_request_id).info(
                ('action', 'finished'),
                ('pack_name', pack_name),
                ('times', times),
                ('total', total),
                ('passed', total - failed),
                ('failed', failed)
            )
        except Exception as e:
            logger(request_id=x_request_id).error(
                ('action', 'finished'),
                ('pack_name', pack_name),
                ('error', str(e)),
                ('tb', traceback.format_exc())
            )
