import json
import traceback
from argparse import ArgumentParser
import logging

import mail.unistat.cpp.cython.logs as logs
import yaml

import mail.borador.unistat.cpp.run as unistat


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('--host')
    parser.add_argument('--port', type=int)
    parser.add_argument('--log', help='log for unistat service')
    parser.add_argument('--source', help='path for tskv log')
    return parser.parse_args()


class ErrorMeter(object):
    def __init__(self):
        self.errors = 0

    def increase(self):
        self.errors += 1

    def update(self, _):
        pass

    def get(self):
        return json.dumps(['unistat_errors_ammm', self.errors])


class BaseMetric(object):
    def __init__(self, errors):
        self.data = {}
        self.errors = errors

    def _update(self, record):
        pass

    def update(self, record):
        try:
            self._update(record=record)
        except Exception as e:
            traceback.print_exc()
            log.error(msg='Something bad happened: record: {0}, error {1}, data {2}'.format(str(record),
                                                                                            str(e), str(self.data)))
            self.errors.increase()

    def get(self):
        return ','.join([json.dumps([k + '_ammm', v]) for k, v in self.data.items()])


class RunningPacks(BaseMetric):
    def _update(self, record):
        if 'action' in record and record['action'] == 'starting' and 'error' not in record:
            self.data[record['pack_name'] + '_running'] = 1
        if 'action' in record and record['action'] == 'finished' and 'error' not in record:
            self.data[record['pack_name'] + '_running'] = 0


class InitEvents(BaseMetric):
    def _update(self, record):
        if 'action' in record and record['action'] == 'init':
            if 'borador_init' not in self.data:
                self.data['borador_init'] = 1
            else:
                self.data['borador_init'] += 1


class PackInfo(BaseMetric):
    def _update(self, record):
        if 'action' in record and record['action'] == 'finished' and 'error' not in record:
            total = int(record['total'])
            passed = int(record['passed'])
            failed = int(record['failed'])
            times = int(record['times'])
            name = record['pack_name']

            self.data[name + '_total'] = total
            self.data[name + '_passed'] = passed
            self.data[name + '_failed'] = failed
            self.data[name + '_times'] = times

            if passed + failed != total:
                log.error(msg='Something bad happened: total={0} passed={1} failed={2} name={3}'
                          .format(total, passed, failed, name))


class AccessEvents(BaseMetric):
    def _update(self, record):
        if 'action' in record and record['action'] == 'access' and '/launch' in record['url']:
            status = record['status']

            if status not in ['200', '400', '500']:
                log.error(msg='Strange status code for /launch: {0}'.format(status))
                self.errors.increase()
            else:
                name = 'launch_' + status
                if name not in self.data:
                    self.data[name] = 1
                else:
                    self.data[name] += 1


def read_config(args):
    with open(args.config) as f:
        return yaml.full_load(f)['config']


def main():
    parser = ArgumentParser()
    parser.add_argument('--config', help='path to config')
    config = read_config(parser.parse_args())

    error_metric = ErrorMeter()
    metrics = [
        error_metric,
        PackInfo(errors=error_metric),
        AccessEvents(errors=error_metric),
        RunningPacks(errors=error_metric),
        InitEvents(errors=error_metric),
    ]

    tskv_log = logs.TskvLog(
        metrics, [],
        False,
        config['source']
    )

    unistat.run(config['host'], config['port'], [tskv_log], config['log'], logLevel='info')


if __name__ == '__main__':
    main()
