# -*- coding: utf-8 -*-
from mail.bots.dev_bot.lib.telegram_api import TelegramApi
from mail.bots.dev_bot.lib.bot import Bot
from mail.bots.dev_bot.lib.db import DB
import mail.bots.dev_bot.lib.settings as settings
import tornado
from tornado import web, ioloop, queues, gen
import argparse
import signal
import logging
import json
import datetime

error_log = logging.getLogger('Error')
service_log = logging.getLogger('Service')
bot = None


def _parse_args():
    parser = argparse.ArgumentParser(add_help=True, description='telegram bot')
    parser.add_argument('-c', '--config', type=str, required=True, help='path to telegram bot config')
    return parser.parse_args()


def signal_handler(signum, frame):
    ioloop.IOLoop.instance().add_callback_from_signal(stop_ioloop)


def stop_ioloop():
    ioloop.IOLoop.instance().stop()
    service_log.info("Finishing bot...")


class Ping(web.RequestHandler):
    def get(self):
        self.write("pong")


class NotifyYtTask(web.RequestHandler):
    @gen.coroutine
    def post(self):
        chat_id = self.get_argument('chat_id')
        exit_code = tornado.escape.json_decode(self.request.body)[0]
        yield bot.api.send_message(chat_id, '[YT] Detemple process finished with code {}'.format(exit_code))


class NotifyYtOp(web.RequestHandler):
    @gen.coroutine
    def post(self):
        chat_id = self.get_argument('chat_id')
        op_type, op_url, op_state = tornado.escape.json_decode(self.request.body)
        yield bot.api.send_message(chat_id, '[YT] Operation {} {} finished with status {}'.format(op_type, op_url, op_state))


class QloudDeployHook(web.RequestHandler):
    """After-deploy hook for qloud"""
    @gen.coroutine
    def post(self):
        res = tornado.escape.json_decode(self.request.body)
        ts = res['statusChangeTs']
        res.pop('statusChangeTs')
        res['date'] = str(datetime.datetime.fromtimestamp(ts/1000))
        environment_id = res['environmentId']
        service_log.debug("GET Deploy notification: {}".format(json.dumps(res)))
        for project_name, chat_ids in bot.deploy_monitoring_projects.iteritems():
            if environment_id.startswith('mail.' + project_name.lower()):
                for _, chat_id, statuses in chat_ids:
                    if res['status'] in statuses:
                        yield bot.api.send_message(chat_id, '\n'.join(k + ' ' + str(v) for k, v in res.iteritems()))


def main():
    global bot
    args = _parse_args()
    settings.init_settings(args.config)
    signal.signal(signal.SIGINT, signal_handler)

    queue = queues.Queue()
    api = TelegramApi(queue, settings.telegram_oauth_token)

    db = DB(settings.conf['DB']['name'])
    db.create_sb_monitoring_users_table()
    db.create_deploy_users_table()

    bot = Bot(queue, api, db, settings.sandbox_oauth_token,
              settings.staff_oauth_token, settings.shinger_print_info,
              settings.mbody_info, settings.template_identifier_info,
              settings.detemple_info,
              settings.staff_auth_groups)
    # Collect users from db, which subscribe to monitoring their sb tasks
    bot.init_monitorings()

    application = web.Application([
        (r"/ping", Ping),
        (r"/notify_yt_op", NotifyYtOp),
        (r"/notify_yt_task", NotifyYtTask),
        (r"/qloud-deployhook", QloudDeployHook),
    ])
    application.listen(settings.conf['Daemon']['port'])

    io_loop = ioloop.IOLoop.instance()

    for handler_number in xrange(settings.conf['Daemon']['concurency']):
        # Handlers which executes user commands
        io_loop.spawn_callback(bot.handler, handler_number)

    # Handler which collect new telegram messages
    io_loop.spawn_callback(api.update_worker)
    io_loop.start()

if __name__ == '__main__':
    main()
