from tornado.httpclient import HTTPError, AsyncHTTPClient
from tornado import gen
from tornado.httputil import url_concat
import json
import logging
from copy import copy

error_log = logging.getLogger('Error')
service_log = logging.getLogger('Service')


def retries(fn):
    max_retries = 3

    @gen.coroutine
    def wrapper(*args, **kwargs):
        retry_num = 0
        while retry_num < max_retries:
            try:
                res = yield fn(*args, **kwargs)
                raise gen.Return(res)
            except HTTPError as e:
                error_log.error("HTTP error while executing request with url: {} headers: {} method: {} error: {}".format(
                    kwargs.get('url'), kwargs.get('headers'), kwargs.get('method'), str(e)))
            finally:
                yield gen.sleep(0.5)
                retry_num += 1
        raise Exception('Max number of attempts is reached with url: {}'.format(kwargs.get('url')))

    return wrapper


class AsyncHttpClient(object):
    def __init__(self, base_url, auth=None, connect_timeout=1, request_timeout=2.5):
        self._http_client = AsyncHTTPClient(force_instance=True,
                                            defaults=dict(connect_timeout=connect_timeout, request_timeout=request_timeout))
        self._base_url = base_url
        self._headers = {}
        if auth:
            self._headers['Authorization'] = "OAuth " + auth

    @property
    def url(self):
        return self._base_url

    @url.setter
    def url(cls, value):
        cls._base_url = value

    @gen.coroutine
    def get(self, path, params):
        url = url_concat('/'.join([self._base_url, path]), params)
        res = yield self.__request(url=url, method='GET', headers=self._headers)
        raise gen.Return(res)

    @gen.coroutine
    def post(self, path, body):
        url = '/'.join([self._base_url, path])
        headers = copy(self._headers)
        headers.update({'Content-Type': 'application/json; charset=utf-8'})
        res = yield self.__request(url=url, method='POST', headers=headers, body=body)
        raise gen.Return(res)

    @gen.coroutine
    def put(self, path, body):
        url = '/'.join([self._base_url, path])
        headers = copy(self._headers)
        headers.update({'Content-Type': 'application/json; charset=utf-8'})
        res = yield self.__request(url=url, method='PUT', headers=headers, body=body)
        raise gen.Return(res)

    @retries
    @gen.coroutine
    def __request(self, url, headers, method, body=None):
        if body:
            res = yield self._http_client.fetch(url, method=method, body=json.dumps(body), headers=headers)
        else:
            res = yield self._http_client.fetch(url, method=method, headers=headers)
        raise gen.Return(res)
