# -*- coding: utf-8 -*-
from types import (BotCommand, AuthorizationRecord, Message, User, Chat, MessageEntity,
                   all_statuses, active_st, finish_st, qloud_statuses)
from async_http_client import AsyncHttpClient
from utils import (extract_text_after_bot_command, cached_decorator, decode_part,
                   get_text_part, BotException)
import logging
import base64
from tornado import gen, ioloop
import json
from email.parser import Parser
from functools import partial

error_log = logging.getLogger('Error')
service_log = logging.getLogger('Service')


class Bot(object):
    def __init__(self, queue, api, db, sb_token,
                 staff_token, shinger_print_info,
                 mbody_info,
                 template_identifier_info,
                 detemple_info,
                 staff_auth_groups):
        self.tasks_queue = queue
        self.api = api
        self.prj_for_deploy = ['Ava', 'Mops', 'Furita', 'MxnetApply', 'Settings', 'ShingerPrint', 'TemplateIdentifier']
        self.prj_for_deploy_monitoring = ['webmail', 'furita', 'shinger-print', 'template-identifier', 'mops',
                                          'firstline', 'catdog', 'ava', 'akita', 'abook', 'calendar-yt', 'calendar-public',
                                          'reminders']
        self._sandbox_token = sb_token
        self._staff_token = staff_token
        self._sb_api_url = "https://sandbox.yandex-team.ru/api/v1.0"
        self._staff_api_url = "http://staff-api.yandex-team.ru/v3"
        self._shinger_print_info = shinger_print_info
        self._mbody_info = mbody_info
        self._template_identifier_info = template_identifier_info
        self._detemple_info = detemple_info
        self._detemple_url = 'http://{}:{}'.format(self._detemple_info.host, self._detemple_info.port)
        self._staff_auth_groups = staff_auth_groups
        # DB with monitoring subscribers
        self._db = db
        # Project to chat_id mapping for deploy notifications
        self._deploy_monitoring_projects = dict()
        self.bot_commands = [
            BotCommand('/start', self.help_handler, 'Bot start'),
            BotCommand('/help', self.help_handler, 'Prints help'),
            BotCommand('/create_mail_release', self.create_mail_release_handler,
                       'Deploy mail project, usage: /create_mail_release mail_project=mops topic=r3828238.28.02 issues=MAILDEV-666,MAILDEV-777'),
            BotCommand('/stop_sb_task', self.stop_sandbox_task, 'Stop sandbox task usage: /stop_sb_task 28382382'),
            BotCommand('/get_sb_tasks', self.get_sb_tasks, 'Get your last 10 sandbox tasks, usage: /get_sb_tasks'),
            BotCommand('/monitoring_sb_tasks', self.monitoring_sb_tasks,
                       'You will notify if your sandbox task finished, usage: /monitoring_sb_tasks'),
            BotCommand('/unsubscribe_monitoring_sb_tasks', self.unsubscribe_monitoring_sb_tasks,
                       'Unsubscribe monitoring your sandbox tasks'),
            BotCommand('/monitoring_yt_tasks', self.monitoring_yt_tasks,
                       'You will notify if your yt task finished, usage: /monitoring_yt_tasks'),
            BotCommand('/unsubscribe_monitoring_yt_tasks', self.unsubscribe_monitoring_yt_tasks,
                       'You will notify if your yt task finished, usage: /monitoring_yt_tasks'),
            BotCommand('/shinger_print_by_email_text', self.shinger_print_by_email_text,
                       'Get shinger print by email text'),
            BotCommand('/shinger_print_by_uid_mid', self.shinger_print_by_uid_mid,
                       'Get shinger print by uid mid /shinger_print_by_uid_mid uid=1120000000051434 mid=165225811329238648 corp=true'),
            BotCommand('/templates_by_email_text', partial(self.get_templates, mode='email_text'),
                       'Get templates by email text /templates_by_email_text sometext'),
            BotCommand('/templates_by_uid_mid', partial(self.get_templates, mode='uid_mid'),
                       'Get templates by uid and mid, usage: /templates_by_uid_mid uid=1120000000051434 mid=165225811329238648 corp=true'),
            BotCommand('/detemple', self.detemple,
                       'Run detemplaization process by robot sherlock usage: \
/detemple yt_path=//home/sherlock/pipeline/1_percent_sample/daily_messages/2018-04-23 \
final_dest=templates/by_stable_hash skip_existing=true/false'),
            BotCommand('/get_yt_operations', self.get_yt_operations, 'Get your current yt operations running with bot'),
            BotCommand('/add_prj_monitoring_deploy', self.add_prj_monitoring_deploy, 'Monitoring deploy status, from selected, of project in qloud, usage: \
/add_prj_monitoring_deploy project=mops statuses=DEPLOY,REMOVE'),
            BotCommand('/remove_prj_monitoring_deploy', self.remove_prj_monitoring_deploy,
                       'Remove project from deploy monitoring /remove_prj_monitoring_deploy project=mops'),
            BotCommand('/unsubscribe_monitoring_deploy', self.unsubscribe_monitoring_deploy,
                       'Unsubsribe deploy monitoring /unsubscribe_monitoring_deploy'),
            BotCommand('/get_monitoring_deploy_projects', self.get_monitoring_deploy_projects,
                       'List of projects for which you are subscribed to notifications for deploy'),
        ]

    @gen.coroutine
    def init_monitorings(self):
        sb_users = self._db.get_all_users_sb_tasks_monitoring()
        service_log.debug("Sandbox users monitoring: {}".format(sb_users))
        if sb_users:
            for tg_login, chat_id in sb_users:
                auth = yield self._get_authorization(tg_login)
                ioloop.IOLoop.current().spawn_callback(self._monitoring_sb_tasks_loop, chat_id, auth)

        deploy_users = self._db.get_all_users_deploy_monitoring()
        service_log.debug("Deploy project users monitoring: {}".format(deploy_users))
        if deploy_users:
            for tg_login, chat_id, projects in deploy_users:
                for project in json.loads(projects):
                    name, statuses = project['name'], project['statuses']
                    if project['name'] in self._deploy_monitoring_projects:
                        self._deploy_monitoring_projects[name].append([tg_login, chat_id, statuses])
                    else:
                        self._deploy_monitoring_projects[name] = [[tg_login, chat_id, statuses]]
        service_log.debug(
            "Deploy monitoring, project to chat_id mapping: {}".format(json.dumps(self.deploy_monitoring_projects)))

    @property
    def deploy_monitoring_projects(self):
        return self._deploy_monitoring_projects

    def authorization_require(f):
        @gen.coroutine
        def wrapped(inst, message, *args, **kwargs):
            tg_login = message.From.Id
            # check user authorization
            if not (yield inst._check_authorization(message, tg_login)):
                return
            yield f(inst, message, *args, **kwargs)

        return wrapped

    @authorization_require
    @gen.coroutine
    def detemple(self, message):
        tg_login, chat_id = message.From.Id, message.chat.Id
        tokens = self._tokenize(message.text)
        arguments = self.__parse_arguments(tokens)
        # Check all required arguments exists
        if 'yt_path' not in arguments or 'final_dest' not in arguments:
            raise BotException("Invalid command params, yt_path or final_dest not found")

        client = AsyncHttpClient(self._detemple_url)
        res = yield client.get('run_sherlock_step', {'yt_path': arguments['yt_path'], 'chat_id': chat_id,
                                                     'tg_login': tg_login,
                                                     'final_dest': arguments['final_dest'],
                                                     'skip_existing': arguments.get('skip_existing', True)})

        if res.code == 200:
            yield self.api.send_message(chat_id,
                                        'Run detemplaization process, you will notify about all finished yt operations')
        else:
            raise BotException("Error while running sherlock steps: {}".format(res.body))

    @authorization_require
    @gen.coroutine
    def get_yt_operations(self, message):
        tg_login = message.From.Id
        chat_id = message.chat.Id
        ans = []
        client = AsyncHttpClient(self._detemple_url)
        res = yield client.get('get_running_ops', {'tg_login': tg_login})
        res = json.loads(res.body)
        for op_type, op_url, op_state, op_progress in res:
            ans.append('{} {} {}\n{}\n'.format(op_type, op_url, op_state, op_progress))

        if ans:
            yield self.api.send_message(chat_id, '\n'.join(ans))
        else:
            yield self.api.send_message(chat_id, 'No running yt operations')

    @staticmethod
    def get_dist(s1, s2):
        '''
        Calculate Levenshtein distance between two lines
        '''
        s1_sz, s2_sz = len(s1), len(s2)
        dp = [[0 for x in range(s2_sz)] for y in range(s1_sz)]
        dp[0][0] = 0 if s1[0] == s2[0] else 1
        for s2_pos in range(1, s2_sz):
            dp[0][s2_pos] = dp[0][s2_pos - 1] + (not s2[s2_pos] == s1[0])
        for s1_pos in range(1, s1_sz):
            dp[s1_pos][0] = dp[s1_pos - 1][0] + (not s1[s1_pos] == s2[0])
        for s1_pos in range(1, s1_sz):
            for s2_pos in range(1, s2_sz):
                if s1[s1_pos] == s2[s2_pos]:
                    dp[s1_pos][s2_pos] = dp[s1_pos - 1][s2_pos - 1]
                else:
                    dp[s1_pos][s2_pos] = min(dp[s1_pos - 1][s2_pos - 1], dp[s1_pos - 1][s2_pos],
                                             dp[s1_pos][s2_pos - 1]) + 1
        return dp[s1_sz - 1][s2_sz - 1]

    def find_project_for_deploy(self, prj_name, monitoring=False):
        '''
        Searches for a project that is minimal different from what the user entered
        '''
        min_dist = 1e9
        best_prj = None
        if monitoring:
            projects = self.prj_for_deploy_monitoring
        else:
            projects = self.prj_for_deploy

        for prj in projects:
            dist = Bot.get_dist(prj, prj_name)
            if dist < min_dist and dist <= 2:
                min_dist = dist
                best_prj = prj
        return best_prj

    def _extract_commands_from_query(self, msg):
        list_of_commands = []
        for entity in msg.entities:
            if entity.Type == 'bot_command':
                offset = entity.offset
                length = entity.length
                list_of_commands.append(msg.text[offset:offset + length])
        return list_of_commands

    def _tokenize(self, text):
        tokens = text.split(' ')
        tokens = filter(lambda x: not x.startswith('/'), tokens)
        return tokens

    def _parse_message(self, data):
        user = User(Id=data['message']['from']['username'],
                    is_bot=data['message']['from']['is_bot'],
                    first_name=data['message']['from']['first_name'],
                    last_name=data['message']['from']['last_name'],
                    username=data['message']['from']['username'])

        chat = Chat(Id=data['message']['chat']['id'],
                    Type=data['message']['chat']['type'])

        msgEntities = []
        if 'entities' in data['message']:
            for entity in data['message']['entities']:
                e = MessageEntity(offset=entity['offset'],
                                  length=entity['length'],
                                  Type=entity['type'])
                msgEntities.append(e)
        msg = Message(message_id=data['message']['message_id'],
                      From=user,
                      date=data['message']['date'],
                      chat=chat,
                      text=data['message']['text'],
                      entities=msgEntities)
        return msg

    @gen.coroutine
    def handler(self, handler_number):
        while True:
            try:
                data = yield self.tasks_queue.get()
                service_log.debug("handler {} start process task {}".format(handler_number, str(data)))
                msg = self._parse_message(data)

                commands = self._extract_commands_from_query(msg)
                if len(commands) == 1:
                    bot_commands = filter(lambda x: x.name == commands[0], self.bot_commands)
                    if len(bot_commands) == 1:
                        yield bot_commands[0].handler(msg)
                    elif len(bot_commands) == 0:
                        yield self.api.send_message(msg.chat.Id, "Sorry i don't know this command")
                elif len(commands) > 1:
                    yield self.api.send_message(msg.chat.Id,
                                                "Sorry don't know how to process more than one command per request")
                elif len(commands) == 0:
                    yield self.api.send_message(msg.chat.Id, "Please choose command")
            except BotException as e:
                error_log.error("Error {}".format(str(e)))
                yield self.api.send_message(msg.chat.Id, str(e))
            except Exception as e:
                error_log.error("Error: {}".format(str(e)))
                yield self.api.send_message(msg.chat.Id, "Sorry I could not execute your command try again later")

    def __parse_arguments(self, tokens):
        s = ' '.join(tokens)
        prev_s = None
        # Remove external spaces
        while prev_s != s:
            prev_s = s
            s = s.replace(' =', '=')
            s = s.replace('= ', '=')
            s = s.replace('  ', ' ')
        clear_tokens = filter(lambda x: '=' in x, s.split(' '))
        service_log.debug("Clear tokens: {}".format(str(clear_tokens)))
        return dict(token.split('=') for token in clear_tokens)

    @authorization_require
    @gen.coroutine
    def create_mail_release_handler(self, message):
        tg_login = message.From.Id
        auth = yield self._get_authorization(tg_login)
        tokens = self._tokenize(message.text)
        arguments = self.__parse_arguments(tokens)
        if not arguments.get('mail_project'):
            msg = 'If you want create release for your project you need specify his name after bot command'
            msg += ' and release topic(optional)'
            msg += ' e.g /create_mail_release mail_project=mops topic=r23283823 issues=MAILDEV-666,MAILDEV-777'
            yield self.api.send_message(message.chat.Id, msg)
            return

        prj = self.find_project_for_deploy(arguments.get('mail_project'))
        if not prj:
            yield self.api.send_message(message.chat.Id, "Sorry i don't know project which you want to release")
            return

        sb_client = AsyncHttpClient(self._sb_api_url, self._sandbox_token)
        issues = arguments.get('issues', '').split(',')
        body = {'owner': 'MAIL', 'type': 'CREATE_MAIL_RELEASE', 'notifications': [{
            "recipients": [
                auth.name
            ],
            "transport": "email",
            "statuses": [
                "SUCCESS",
                "FAILURE",
                "EXCEPTION"
            ]
        }], 'custom_fields': [{'name': 'mail_project', 'value': prj},
                              {'name': 'topic', 'value': arguments.get('topic')},
                              {'name': 'issues', 'value': issues}]}

        res = yield sb_client.post('task', body)
        _id = json.loads(res.body)['id']
        yield sb_client.put('batch/tasks/start', {"id": [_id]})

        service_log.debug("Create mail release uid: {} response: {}".format(tg_login, res.body))
        ans = "Task start successfuly {url}".format(url="https://sandbox.yandex-team.ru/task/{}/view".format(_id))
        yield self.api.send_message(message.chat.Id, ans)

    @authorization_require
    @gen.coroutine
    def stop_sandbox_task(self, message):
        tokens = self._tokenize(message.text)
        if len(tokens) != 1:
            yield self.api.send_message(message.chat.Id, 'Specify task which you want to stop')
            return
        task_id = tokens[0]
        sb_client = AsyncHttpClient(self._sb_api_url, self._sandbox_token)
        res = yield sb_client.put('batch/tasks/stop', {"id": [task_id]})

        service_log.debug("Stop task result: {}".format(json.loads(res.body)[0]['message']))
        ans = json.loads(res.body)[0]['message']
        yield self.api.send_message(message.chat.Id, str(ans))

    @authorization_require
    @gen.coroutine
    def get_sb_tasks(self, message):
        tg_login = message.From.Id
        sb_client = AsyncHttpClient(self._sb_api_url, self._sandbox_token)
        res = yield sb_client.get('task', {'limit': 10, 'order': '-time',
                                           'author': (yield self._get_authorization(tg_login)).name})

        service_log.debug("Get sb tasks response code: {}".format(str(res.code)))
        ans = ''
        for task in json.loads(res.body)['items']:
            status = task['status']
            Id = task['id']
            url = "https://sandbox.yandex-team.ru/task/{}/view".format(Id)
            ans += "Task: {} {}\n".format(url, status)
        yield self.api.send_message(message.chat.Id, ans)

    @authorization_require
    @gen.coroutine
    def monitoring_sb_tasks(self, message):
        tg_login = message.From.Id
        if self._db.get_user_sb_tasks_monitoring(tg_login):
            yield self.api.send_message(message.chat.Id, "You already subscribe to sb tasks monitoring")
            return

        self._db.add_user_to_sb_tasks_monitoring(tg_login, message.chat.Id)
        auth = yield self._get_authorization(tg_login)
        ioloop.IOLoop.current().spawn_callback(self._monitoring_sb_tasks_loop, message.chat.Id, auth)
        yield self.api.send_message(message.chat.Id, "You success subscribe to notifications about your sandbox tasks")

    @authorization_require
    @gen.coroutine
    def add_prj_monitoring_deploy(self, message):
        tg_login, chat_id = message.From.Id, message.chat.Id

        tokens = self._tokenize(message.text)
        arguments = self.__parse_arguments(tokens)
        project = arguments.get('project')
        statuses = arguments.get('statuses')
        if project:
            user = self._db.get_user_deploy_monitoring(tg_login)
            real_project = self.find_project_for_deploy(project, True)
            if not real_project:
                raise BotException("Unknown project {} allowed {}".format(project, str(self.prj_for_deploy_monitoring)))
            if statuses:
                st = []
                for status in statuses.split(','):
                    if status not in qloud_statuses:
                        raise BotException("Unknown status {} allowed {}".format(status, str(qloud_statuses)))
                    else:
                        st.append(status)
            else:
                st = qloud_statuses
            res = [{'name': real_project, 'statuses': st}]
            if user:
                projects = filter(lambda x: x['name'] != real_project, json.loads(user[2]))
                projects.extend(res)
            else:
                projects = res

            self._db.remove_user_from_deploy_monitoring(tg_login)
            self._db.add_user_to_deploy_monitoring(tg_login, chat_id, json.dumps(projects))

            if real_project in self.deploy_monitoring_projects:
                self.deploy_monitoring_projects[real_project] = filter(lambda x: x[0] != tg_login,
                                                                       self.deploy_monitoring_projects[real_project])
                self.deploy_monitoring_projects[real_project].append([tg_login, chat_id, st])
            else:
                self.deploy_monitoring_projects[real_project] = [[tg_login, chat_id, st]]

            yield self.api.send_message(message.chat.Id,
                                        "Success subscribe to notify deploy project: {} statuses: {}".format(
                                            real_project, st))
        else:
            raise BotException(
                "Please list the projects for which you want to receive notification of the status of the deploy")

    @authorization_require
    @gen.coroutine
    def remove_prj_monitoring_deploy(self, message):
        tg_login, chat_id = message.From.Id, message.chat.Id

        tokens = self._tokenize(message.text)
        arguments = self.__parse_arguments(tokens)
        project = arguments.get('project')
        if project:
            user = self._db.get_user_deploy_monitoring(tg_login)
            real_project = self.find_project_for_deploy(project, True)
            if not real_project:
                raise BotException("Unknown project {} allowed {}".format(project, str(self.prj_for_deploy_monitoring)))
            if user:
                projects = filter(lambda x: x['name'] != real_project, json.loads(user[2]))
            else:
                raise BotException("You must add at least one project to monitor to start deleting them")

            self._db.remove_user_from_deploy_monitoring(tg_login)
            self._db.add_user_to_deploy_monitoring(tg_login, chat_id, json.dumps(projects))
            self.deploy_monitoring_projects[real_project] = filter(lambda x: x[0] != tg_login,
                                                                   self.deploy_monitoring_projects[real_project])
            yield self.api.send_message(message.chat.Id, "Success remove project from deploy monitoring")
        else:
            raise BotException(
                "Please list the projects for which you want to receive notification of the status of the deploy")

    @authorization_require
    @gen.coroutine
    def unsubscribe_monitoring_deploy(self, message):
        tg_login = message.From.Id
        check_user = self._db.get_user_deploy_monitoring(tg_login)
        if not check_user:
            yield self.api.send_message(message.chat.Id, "You are not subscribe to deploy monitoring")
            return

        self._db.remove_user_from_deploy_monitoring(tg_login)
        yield self.api.send_message(message.chat.Id, "Success unsubsribe from deploy monitoring")

    @authorization_require
    @gen.coroutine
    def get_monitoring_deploy_projects(self, message):
        tg_login = message.From.Id
        check_user = self._db.get_user_deploy_monitoring(tg_login)
        if check_user:
            projects = json.loads(check_user[2])
            res = '\n'.join([prj['name'] + ' ' + ','.join(prj['statuses']) for prj in projects])
            if res:
                yield self.api.send_message(message.chat.Id, res)
            else:
                yield self.api.send_message(message.chat.Id, 'You do not have projects that you monitoring deploy')
        else:
            yield self.api.send_message(message.chat.Id, "You have no projects to monitoring deploy")

    def _check_notifications(self, notifications, auth):
        for notification in notifications:
            if auth.name in notification['recipients']:
                return True
        return False

    @gen.coroutine
    def _monitoring_sb_tasks_loop(self, chat_id, auth):
        sb_client = AsyncHttpClient(self._sb_api_url, self._sandbox_token)
        res_auth, res_robot = yield [
            sb_client.get('task', {'limit': 20, 'status': ','.join(active_st), 'author': auth.name}),
            sb_client.get('task', {'limit': 20, 'status': ','.join(active_st), 'author': 'robot-gerrit'})]
        filtered_robot_tasks = filter(lambda item: self._check_notifications(item['notifications'], auth),
                                      json.loads(res_robot.body)['items'])
        last_tasks = json.loads(res_auth.body)['items'] + filtered_robot_tasks
        actual_tasks_info = []
        for task in last_tasks:
            actual_tasks_info.append([task['id'], task['status']])

        while self._db.get_user_sb_tasks_monitoring(auth.tg_login):
            nxt = gen.sleep(5)
            try:
                res_auth, res_robot = yield [
                    sb_client.get('task', {'limit': 20, 'status': ','.join(all_statuses), 'author': auth.name}),
                    sb_client.get('task', {'limit': 20, 'status': ','.join(all_statuses), 'author': 'robot-gerrit'})]
                filtered_robot_tasks = filter(lambda item: self._check_notifications(item['notifications'], auth),
                                              json.loads(res_robot.body)['items'])
                last_tasks = json.loads(res_auth.body)['items'] + filtered_robot_tasks
                last_tasks_info = []
                for task in last_tasks:
                    last_tasks_info.append([task['id'], task['status']])

                actual_tasks_info = yield self._get_new_tasks_list(last_tasks_info, actual_tasks_info, chat_id)
            except Exception as e:
                error_log.error("Error while monitoring sb tasks: {}".format(str(e)))
            finally:
                yield nxt

    @gen.coroutine
    def _get_new_tasks_list(self, last_tasks_info, actual_tasks_info, chat_id):
        new_actual_tasks_info = []
        for last_task_id, last_task_status in last_tasks_info:
            is_change_status = False
            for actual_task_id, actual_task_status in actual_tasks_info:
                if last_task_id == actual_task_id and last_task_status != actual_task_status and last_task_status in finish_st:
                    is_change_status = True
            if last_task_status in active_st:
                new_actual_tasks_info.append([last_task_id, last_task_status])
            if is_change_status:
                task_url = "https://sandbox.yandex-team.ru/task/{}/view".format(last_task_id)
                status = last_task_status
                yield self.api.send_message(chat_id, "Task {} finished with status {}".format(task_url, status))

        raise gen.Return(new_actual_tasks_info)

    @authorization_require
    @gen.coroutine
    def unsubscribe_monitoring_sb_tasks(self, message):
        tg_login = message.From.Id
        check_user = self._db.get_user_sb_tasks_monitoring(tg_login)
        if not check_user:
            yield self.api.send_message(message.chat.Id, "You are not subscribe to monitor sandbox tasks")
            return

        self._db.remove_user_from_sb_tasks_monitoring(tg_login)
        yield self.api.send_message(message.chat.Id, "Success unsubsribe")

    @authorization_require
    @gen.coroutine
    def unsubscribe_monitoring_yt_tasks(self, message):
        yield self.api.send_message(message.chat.Id, "This is not implemented, sorry")

    @authorization_require
    @gen.coroutine
    def monitoring_yt_tasks(self, message):
        # yt.list_operations(user="odin", cursor_time=datetime.utcnow(), cursor_direction="past", limit=3)["operations"]
        yield self.api.send_message(message.chat.Id, "This is not implemented, sorry")

    @gen.coroutine
    def _get_shinger_print(self, text):
        url = 'http://{}:{}'.format(self._shinger_print_info.host, self._shinger_print_info.port)
        body = {'text': text}
        client = AsyncHttpClient(url)
        r = yield client.post('/', body=body)
        raise gen.Return(json.loads(r.body))

    @gen.coroutine
    def __shinger_print_by_email_text(self, message, bot_command='/shinger_print_by_email_text'):
        text = extract_text_after_bot_command(bot_command, message)
        shingers = yield self._get_shinger_print(text)
        raise gen.Return(shingers)

    @authorization_require
    @gen.coroutine
    def shinger_print_by_email_text(self, message):
        shingers = yield self.__shinger_print_by_email_text(message)
        yield self.api.send_message(message.chat.Id, json.dumps(shingers))

    @gen.coroutine
    def __shinger_print_by_uid_mid(self, message):
        tokens = self._tokenize(message.text)
        arguments = self.__parse_arguments(tokens)
        if 'uid' not in arguments or 'mid' not in arguments:
            raise BotException("Invalid command params, mid or uid not found")

        if arguments.get('corp') != 'true':
            mbody_url = 'http://{}:{}'.format(self._mbody_info.prestable_host, self._mbody_info.prestable_port)
        else:
            mbody_url = 'http://{}:{}'.format(self._mbody_info.intranet_prestable_host,
                                              self._mbody_info.intranet_prestable_port)

        client = AsyncHttpClient(mbody_url)
        try:
            res = yield client.get('message_source', {'uid': arguments['uid'], 'mid': arguments['mid']})
        except:
            raise BotException("Can't find requested message")

        body = json.loads(res.body)
        text = base64.b64decode(body['text'])
        service_log.debug("Decode text part: {}".format(text))
        text = get_text_part(Parser().parsestr(text))
        charset = text.get_content_charset()
        content = decode_part(text.get_payload(), text.get('Content-Transfer-Encoding'))
        content = content.decode(charset).encode('utf8')
        service_log.debug("Content {}".format(json.dumps(content)))
        shingers = yield self._get_shinger_print(content)
        raise gen.Return(shingers)

    @authorization_require
    @gen.coroutine
    def shinger_print_by_uid_mid(self, message):
        shingers = yield self.__shinger_print_by_uid_mid(message)
        yield self.api.send_message(message.chat.Id, json.dumps(shingers))

    @authorization_require
    @gen.coroutine
    def get_templates(self, message, mode):
        if mode == 'uid_mid':
            shingers = yield self.__shinger_print_by_uid_mid(message)
        elif mode == 'email_text':
            shingers = yield self.__shinger_print_by_email_text(message, bot_command='/templates_by_email_text')
        else:
            raise ValueError("Unknown get templates mode")
        if not shingers:
            yield self.api.send_message(message.chat.Id, '[]')
            return

        service_log.debug("get_templates shingers {}".format(str(shingers)))
        tics_url = 'http://{}:{}'.format(self._template_identifier_info.host, self._template_identifier_info.port)
        client = AsyncHttpClient(tics_url)
        res = yield client.get('match', {'hashes': ','.join(map(lambda x: str(x), shingers))})
        yield self.api.send_message(message.chat.Id, res.body)

    @cached_decorator
    @gen.coroutine
    def _get_authorization(self, tg_login):
        client = AsyncHttpClient(self._staff_api_url, self._sandbox_token)
        res_arr = yield [client.get('persons', {'accounts.type': 'telegram', 'accounts.value': tg_login,
                                                'groups.group.parent.department.url': group})
                         for group in self._staff_auth_groups]
        for auth_res in res_arr:
            service_log.debug("Authorization result: {} {}".format(auth_res.body, auth_res.code))
            if auth_res.code == 200:
                res = json.loads(auth_res.body)['result']
                if len(res) == 1 and not res[0].get('is_deleted'):
                    raise gen.Return(AuthorizationRecord(tg_login, res[0].get('login')))
        raise gen.Return(AuthorizationRecord(tg_login))

    @gen.coroutine
    def _check_authorization(self, message, tg_login):
        auth = yield self._get_authorization(tg_login)
        if not auth.name:
            yield self.api.send_message(message.chat.Id, "I'm sorry I do not know you, try to sign in")
            raise gen.Return(False)
        raise gen.Return(True)

    @authorization_require
    @gen.coroutine
    def help_handler(self, message):
        res = '\n'.join(command.name + " " + command.description for command in self.bot_commands)
        yield self.api.send_message(message.chat.Id, res)
