#pragma once

#include "account_provider.h"
#include <common/json.h>
#include <common/server_info.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/find.h>
#include <yplatform/module.h>

namespace botserver::auth {

struct blackbox_module
    : account_provider
    , yplatform::module
{
    shared_ptr<yhttp::cluster_client> http_client;

    blackbox_module(yplatform::reactor& reactor, yplatform::ptree conf)
        : http_client(make_shared<yhttp::cluster_client>(reactor, conf.get_child("http")))
    {
    }

    future<optional<mail_account>> get_mail_account(
        task_context_ptr ctx,
        botpeer /*botpeer*/,
        string email) override
    {
        promise<optional<mail_account>> promise;
        auto handler = std::bind(
            &blackbox_module::handle_response, shared_from(this), promise, ph::_1, ph::_2);
        auto request = make_request(email);
        http_client->async_run(ctx, request, handler);
        return promise;
    }

    yhttp::request make_request(string email)
    {
        auto request = yhttp::request::GET(
            "/blackbox" +
            yhttp::url_encode({ { "method", "userinfo" },
                                { "format", "json" },
                                { "emails", "testone" },
                                { "addrtotest", email },
                                { "login", email },
                                { "userip", my_ip },
                                { "user_port", my_port } }));
        return request;
    }

    void handle_response(
        promise<optional<mail_account>> promise,
        error_code err,
        yhttp::response response)
    {
        if (err) return set_exception(promise, err.message());
        if (response.status != 200)
        {
            return set_exception(promise, "bad status " + std::to_string(response.status));
        }
        json_value json;
        if (auto error = json.parse(response.body)) return set_exception(promise, *error);
        if (json.has_member("error")) return set_exception(promise, json["error"].to_string());

        auto user = first_array_element(json["users"]);
        if (!user) return promise.set({});

        auto addr = first_array_element((*user)["address-list"]);
        if (!addr) return promise.set({});

        auto uid = (*user)["uid"]["value"].to_string();
        auto email = (*addr)["address"].to_string();
        if (uid.empty() || email.empty()) return promise.set({});
        promise.set(mail_account{ uid, email });
    }

    template <typename Promise>
    void set_exception(Promise promise, string error_message)
    {
        promise.set_exception(runtime_error("auth error: " + error_message));
    }

    optional<json_value> first_array_element(json_value array)
    {
        if (array.size() < 1) return {};
        return *array.array_begin();
    }
};

}
