#pragma once

#include "account_provider.h"
#include <common/botpeer.h>
#include <common/json.h>
#include <ymod_httpclient/cluster_client.h>
#include <ymod_httpclient/url_encode.h>
#include <yplatform/find.h>
#include <yplatform/module.h>
#include <boost/algorithm/string.hpp>

namespace botserver::auth {

struct yandex_staff_module
    : account_provider
    , yplatform::module
{
    shared_ptr<yhttp::cluster_client> http_client;
    string oauth_token;

    yandex_staff_module(yplatform::reactor& reactor, yplatform::ptree conf)
        : http_client(make_shared<yhttp::cluster_client>(reactor, conf.get_child("http")))
        , oauth_token(conf.get<string>("oauth_token"))
    {
    }

    future<optional<mail_account>> get_mail_account(
        task_context_ptr ctx,
        botpeer botpeer,
        string email) override
    {
        promise<optional<mail_account>> promise;
        auto handler = std::bind(
            &yandex_staff_module::handle_response, shared_from(this), promise, ph::_1, ph::_2);
        auto request = make_request(botpeer, email);
        http_client->async_run(ctx, request, handler);
        return promise;
    }

    yhttp::request make_request(botpeer botpeer, string email)
    {
        auto request = yhttp::request::GET(
            "/v3/persons" +
                yhttp::url_encode({ { "telegram_accounts.value_lower",
                                      boost::algorithm::to_lower_copy(botpeer.username) },
                                    { "work_email", email },
                                    { "official.is_dismissed", "false" },
                                    { "_fields", "uid,work_email" },
                                    { "_one", 1 } }),
            "Authorization: OAuth " + oauth_token + "\r\n");
        return request;
    }

    void handle_response(
        promise<optional<mail_account>> promise,
        error_code err,
        yhttp::response response)
    {
        if (err) return set_exception(promise, err.message());
        if (response.status == 404) return promise.set({});
        if (response.status != 200)
        {
            return set_exception(promise, "bad status " + std::to_string(response.status));
        }
        json_value json;
        if (auto error = json.parse(response.body)) return set_exception(promise, *error);
        auto email = json["work_email"].to_string();
        auto uid = json["uid"].to_string();
        promise.set(mail_account{ uid, email });
    }

    template <typename Promise>
    void set_exception(Promise promise, string error_message)
    {
        promise.set_exception(runtime_error("auth error: " + error_message));
    }
};

}
