#pragma once

#include "interface.h"
#include "pq_args.h"
#include "pq_converters.h"
#include <common/types.h>
#include <ymod_pq/call.h>
#include <yplatform/module.h>
#include <boost/random/random_device.hpp>
#include <boost/random/discrete_distribution.hpp>
#include <cmath>

namespace botserver::db {

struct settings
{
    string conninfo;
    bool log_timins;
    struct
    {
        using codes_range_t = std::pair<uint64_t, uint64_t>;
        const unsigned max_code_length = 19;

        unsigned code_length;
        codes_range_t codes_range;
        unsigned ttl_sec;
    } otp;

    void parse_ptree(yplatform::ptree conf)
    {
        conninfo = conf.get<string>("conninfo");
        log_timins = conf.get<bool>("log_timins");
        otp.code_length = conf.get<unsigned>("otp.code_length");
        if (otp.code_length > otp.max_code_length)
        {
            throw std::runtime_error("code_length is too big, integer overflow");
        }
        otp.codes_range = { 0, static_cast<uint64_t>(std::pow(10, otp.code_length)) - 1 };
        otp.ttl_sec = conf.get<unsigned>("otp.ttl_sec");
    }
};

inline settings make_settings(const yplatform::ptree& conf)
{
    struct settings res;
    res.parse_ptree(conf);
    return res;
}

template <typename PqModule, typename Random>
struct module_impl
    : links
    , otp
    , yplatform::module
{
    settings settings;
    shared_ptr<PqModule> pq;
    Random random;

    module_impl(struct settings settings)
        : settings(settings), pq(find_module<PqModule>("botdb_pq"))
    {
    }

    module_impl(yplatform::ptree conf) : module_impl(make_settings(conf))
    {
    }

    void add(task_context_ptr ctx, link link) override
    {
        auto args = make_pq_args(
            link.mail_account.uid,
            link.mail_account.email,
            link.botpeer.chat_id,
            link.botpeer.bot_id,
            string("telegram")); // XXX magic enum
        pq_request(ctx, "create_link", args).get();
    }

    optional<link> lookup(task_context_ptr ctx, botpeer botpeer) override
    {
        auto args =
            make_pq_args(botpeer.chat_id, botpeer.bot_id, string("telegram")); // XXX magic enum
        auto converter = make_mail_accounts_converter();
        pq_request(ctx, "lookup_mail_account", args, converter).get();

        if (converter->result.empty())
        {
            return {};
        }
        if (converter->result.size() == 1)
        {
            return link{ botpeer, converter->result.front() };
        }
        throw std::runtime_error("multiple mail accounts for botpeer");
    }

    string gen_code(task_context_ptr ctx, botpeer botpeer, mail_account mail_account) override
    {
        auto code = random_code();
        auto args = make_pq_args(
            mail_account.uid,
            mail_account.email,
            botpeer.chat_id,
            botpeer.bot_id,
            string("telegram"), // XXX magic enum
            code);
        pq_request(ctx, "save_otp", args).get();
        return code;
    }

    otp_check_result check_code(task_context_ptr ctx, botpeer botpeer, string code) override
    {
        auto args = make_pq_args(
            botpeer.chat_id,
            botpeer.bot_id,
            string("telegram"), // XXX magic enum
            code,
            settings.otp.ttl_sec);
        auto converter = make_mail_accounts_converter();
        pq_request(ctx, "verify_otp", args, converter, ymod_pq::request_target::master).get();
        if (converter->result.empty())
        {
            return { false, {} };
        }
        if (converter->result.size() == 1)
        {
            return { true, converter->result.front() };
        }
        throw std::runtime_error("multiple mail accounts for botpeer");
    }

    template <typename Converter>
    ymod_pq::future_result pq_request(
        task_context_ptr ctx,
        string request,
        ymod_pq::bind_array_ptr args,
        Converter converter,
        ymod_pq::request_target target = ymod_pq::request_target::try_replica)
    {
        return pq->request(
            ctx,
            settings.conninfo,
            request,
            args,
            converter,
            settings.log_timins,
            yplatform::time_traits::duration::max(),
            target);
    }

    ymod_pq::future_result pq_request(
        task_context_ptr ctx,
        string request,
        ymod_pq::bind_array_ptr args)
    {
        return pq->execute(
            ctx,
            settings.conninfo,
            request,
            args,
            settings.log_timins,
            yplatform::time_traits::duration::max(),
            ymod_pq::request_target::master);
    }

    std::string random_code()
    {
        auto res = random(settings.otp.codes_range.first, settings.otp.codes_range.second);

        std::stringstream ss;
        ss << std::setfill('0') << std::setw(settings.otp.code_length) << res;
        return ss.str();
    }
};

struct random_device_generator
{
    template <typename Int>
    Int operator()(Int min, Int max)
    {
        boost::random::random_device device;
        boost::random::uniform_int_distribution<Int> dist(min, max);
        return dist(device);
    }
};

using module = module_impl<ymod_pq::cluster_call, random_device_generator>;

}
