#pragma once

#include "markdown.h"
#include "common/message_author.h"
#include "common/platforms.h"
#include "common/types.h"
#include <boost/format.hpp>

namespace botserver::i18n {

enum class language
{
    en,
    ru,
};

using boost::format;
using translations = map<language, string>;

namespace detail {

inline string format_author_for_title(message_author author)
{
    if (author.name.empty()) return author.login;
    if (author.login.empty()) return author.name;

    return author.name + " (" + author.login + ")";
}

inline string format_author_for_body(message_author author)
{
    auto name = author.name.size() ? author.name : author.login;
    if (author.profile_link.empty()) return name;
    if (name.empty()) return author.profile_link;

    return name + " (" + author.profile_link + ")";
}

inline string platform(language lang, platform_name paltform)
{
    static const map<platform_name, translations> platform_names{
        {
            {
                platform_name::telegram,
                {
                    { language::ru, "Telegram" },
                    { language::en, "Telegram" },
                },
            },
        },
    };
    return platform_names.at(paltform).at(lang);
}

inline string format_date_time(language lang, time_t received_date)
{
    static const translations formats{
        { language::ru, "%H:%M:%S %d/%m/%y" },
        { language::en, "%H:%M:%S %m/%d/%y" },
    };
    std::stringstream ss;
    ss << std::put_time(std::localtime(&received_date), formats.at(lang).c_str());
    return ss.str();
}

}

inline markdown_string missing_code(language lang)
{
    static const translations translations{
        { language::ru,
          "Чтобы отправить мне код, введите команду `/code`, а затем через пробел код "
          "подтверждения\\." },
        { language::en,
          "To send me code, type `/code` and then the confirmation code, separated by a space\\." },
    };
    return markdown_string(translations.at(lang));
}

inline markdown_string invalid_code_number(language lang)
{
    static const translations translations{
        { language::ru,
          "Это неправильный код\\. Привяжите адрес заново — введите команду `/email`\\." },
        { language::en, "Incorrect code\\. Type `/email` to relink the address\\." },
    };
    return markdown_string(translations.at(lang));
}

inline string otp_rate_limit_exceeded(language lang)
{
    static const translations translations{
        { language::ru, "Превышено количество попыток привязать почту. Попробуйте позже." },
        { language::en,
          "You have exceeded the number of attempts to link the email. Please try again later." },
    };
    return translations.at(lang);
}

inline string email_binded_successfully(language lang)
{
    static const translations translations{
        { language::ru,
          "Почта успешно привязана. Теперь вы можете переслать мне сообщение, и я отправлю его в "
          "ваш ящик." },
        { language::en,
          "Email address successfully linked. Now you can forward me messages, and I'll send them "
          "to your mailbox." },
    };
    return translations.at(lang);
}

inline markdown_string to_start_bind_email_address(language lang)
{
    static const translations translations{
        { language::ru,
          "Чтобы начать, привяжите адрес почты — введите команду `/email`, а затем через пробел "
          "ваш адрес\\." },
        { language::en,
          "To start, link your email address — type `/email` and then your email address, "
          "separated by a space\\." },
    };
    return markdown_string(translations.at(lang));
}

inline markdown_string bind_email_address(language lang)
{
    static const translations translations{
        { language::ru,
          "Вы не привязали адрес почты\\. Чтобы привязать, введите команду `/email`, а затем через "
          "пробел ваш адрес\\." },
        { language::en,
          "You haven't linked an email address\\. To link an address, type `/email`, and then your "
          "email address, separated by a space\\." },
    };
    return markdown_string(translations.at(lang));
}

inline markdown_string missing_email(language lang)
{
    static const translations translations{
        { language::ru,
          "Чтобы привязать почту, введите команду `/email`, а затем через пробел ваш адрес\\." },
        { language::en,
          "To link an address, type `/email`, and then your email address, separated by a "
          "space\\." },
    };
    return markdown_string(translations.at(lang));
}

inline string invalid_email(language lang)
{
    static const translations translations{
        { language::ru, "Вы указали некорректный адрес. Проверьте и введите еще раз." },
        { language::en, "You entered an incorrect address. Check and try again." },
    };
    return translations.at(lang);
}

inline markdown_string send_code_from_email(language lang, string email)
{
    static const translations translations{
        { language::ru,
          "Проверьте почту %1% — на нее отправлен код подтверждения\\. Чтобы отправить его мне, "
          "введите команду `/code`, а затем через пробел код подтверждения\\." },
        { language::en,
          "I sent a confirmation code to %1%\\. Check your email\\. To send it back to me, type "
          "`/code` and then the confirmation code, separated by a space\\." },
    };
    auto escaped_email = markdown_escape(email);
    return markdown_string((format(translations.at(lang)) % escaped_email).str());
}

inline string your_otp_code_for_bot(language lang, string code)
{
    static const translations translations{
        { language::ru, "%1% — ваш код для привязки к боту Яндекс.Почты 360" },
        { language::en, "%1% — Your code for linking Yandex.Mail 360 bot" },
    };
    return (format(translations.at(lang)) % code).str();
}

inline string forward_and_it_will_be_send_to_email(language lang)
{
    static const translations translations{
        { language::ru,
          "Здравствуйте.\nЯ телеграм-бот Яндекс.Почты 360. Перешлите мне важное сообщение, которое "
          "вы не успеваете сейчас прочитать — и я отправлю его в ваш ящик." },
        { language::en,
          "Hi!\nI'm the Yandex.Mail 360 Telegram bot. Forward me an important message that you "
          "don't have time to read right now, and I'll send it to your mailbox." },
    };
    return translations.at(lang);
}

inline string otp_code_is(language lang, string username, string code)
{
    static const translations translations{
        { language::ru,
          "Здравствуйте, %1%!\nВы запросили код подтверждения, чтобы привязать этот "
          "адрес к телеграм-боту Яндекс.Почты 360.\nВот он:\n%2%\nЕсли вы не запрашивали код, "
          "просто проигнорируйте это сообщение." },
        { language::en,
          "Hello, %1%!\nYou've requested a confirmation code to link this address to the "
          "Yandex.Mail 360 Telegram bot.\nYour code is:\n%2%\nIf you didn't request the code, just "
          "ignore this message." },
    };
    return (format(translations.at(lang)) % username % code).str();
}

inline string pong(language lang)
{
    static const translations translations{
        { language::ru, "pong" },
        { language::en, "pong" },
    };
    return translations.at(lang);
}

inline string message_too_big(language lang)
{
    static const translations translations{
        { language::ru, "Сообщение слишком большое. Разбейте его на части и попробуйте снова." },
        { language::en, "The message is too long. Break it into parts and try again." },
    };
    return translations.at(lang);
}

inline string forwarded_from_user_on_messenger(
    language lang,
    message_author forwarded_from,
    platform_name paltform)
{
    static const translations translations{
        { language::ru, "Сообщение из %2% от %1%" },
        { language::en, "%2% message from %1%" },
    };
    auto platform_name = detail::platform(lang, paltform);
    return (format(translations.at(lang)) % detail::format_author_for_title(forwarded_from) %
            platform_name)
        .str();
}

inline string forwarded_from_messenger(language lang, platform_name paltform)
{
    static const translations translations{
        { language::ru, "Сообщение из %1%" },
        { language::en, "%1% message" },
    };
    auto platform_name = detail::platform(lang, paltform);
    return (format(translations.at(lang)) % platform_name).str();
}

inline string received_from_user(
    language lang,
    string text,
    message_author forwarded_from,
    time_t received_date)
{
    static const translations translations{
        { language::ru, "%1% (%2%):\n\n%3%" },
        { language::en, "%1% (%2%):\n\n%3%" },
    };
    auto date_time_str = detail::format_date_time(lang, received_date);
    return (format(translations.at(lang)) % detail::format_author_for_body(forwarded_from) %
            date_time_str % text)
        .str();
}

inline string error_occured(language lang, string id)
{
    static const translations translations{
        { language::ru,
          "При обработке сообщения произошла ошибка, код ошибки: %1%. Напишите в службу поддержки "
          "Яндекс.Почты: https://yandex.ru/support/mail/" },
        { language::en,
          "An error occurred while processing the message. Error code: %1%. Please contact "
          "Yandex.Mail support: https://yandex.ru/support/mail/" },
    };
    return (format(translations.at(lang)) % id).str();
}

inline string service_unavailable(language lang)
{
    static const translations translations{
        { language::ru, "Бот временно недоступен. Попробуйте позже." },
        { language::en, "Bot is currently unavailable. Please try again later." },
    };
    return translations.at(lang);
}

}
