#pragma once

#include <common/mail_message.h>
#include <common/types.h>
#include <yplatform/util/sstream.h>
#include <yplatform/encoding/base64.h>

namespace botserver::mail {

struct composer
{
    const string boundary = "==============6590253278595032364.botserver.mail.yandex.net==";

    string_ptr compose_text_message(mail_message_ptr message, string sender_email)
    {
        auto eml = make_shared<string>();
        yplatform::sstream str(*eml);
        str << "From: " << sender_email << "\r\n"
            << "To: " << message->to_email << "\r\n"
            << "Subject: " << b64_encode_header(message->subject) << "\r\n"
            << "Content-Type: text/plain; charset=utf-8\r\n"
            << "Content-transfer-encoding: base64\r\n"
            << "\r\n"
            << yplatform::base64_encode_str(message->text);
        return eml;
    }

    string_ptr compose_message_with_attachments(mail_message_ptr message, string sender_email)
    {
        auto eml = make_shared<string>();
        yplatform::sstream str(*eml);
        write_multipart_header(str, message, sender_email);
        if (message->text.length())
        {
            str << "\r\n--" << boundary << "\r\n";
            write_text_part(str, message->text);
        }
        for (auto attachment : message->attachments)
        {
            str << "\r\n--" << boundary << "\r\n";
            write_attachment_part(str, attachment);
        }
        str << "\r\n--" << boundary << "--\r\n";
        return eml;
    }

    void write_multipart_header(
        yplatform::sstream& str,
        mail_message_ptr message,
        string sender_email)
    {
        str << "From: " << sender_email << "\r\n"
            << "To: " << message->to_email << "\r\n"
            << "Subject: " << b64_encode_header(message->subject) << "\r\n"
            << "Content-Type: multipart/mixed; boundary=\"" << boundary << "\"\r\n"
            << "\r\n";
    }

    string b64_encode_header(string str)
    {
        return "=?UTF-8?B?" + yplatform::base64_encode_str(str) + "?=";
    }

    void write_text_part(yplatform::sstream& str, string text)
    {
        str << "Content-Type: text/plain; charset=utf-8\r\n"
            << "Content-transfer-encoding: base64\r\n"
            << "\r\n"
            << yplatform::base64_encode_str(text);
    }

    void write_attachment_part(yplatform::sstream& str, mail_attachment attachment)
    {
        str << "Content-Type: " << get_attachment_type(attachment) << "\r\n"
            << "Content-transfer-encoding: base64\r\n"
            << "Content-Disposition: " << get_attachment_disposition(attachment) << "\r\n"
            << "\r\n"
            << yplatform::base64_encode_str(*attachment.content);
    }

    string get_attachment_type(mail_attachment attachment)
    {
        return attachment.mime_type.length() ? attachment.mime_type : "application/octet-stream";
    }

    string get_attachment_disposition(mail_attachment attachment)
    {
        if (attachment.name.length())
        {
            return "attachment; filename=" + attachment.name;
        }
        else
        {
            return "attachment";
        };
    }
};

}